<?php

namespace AppBundle\Command;

use AppBundle\Entity\InscriptionStatus;
use AppBundle\Entity\PaymentStatus;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Command responsável pela verificação da integrações pendentes.
 *
 * @author João Zica <joaozica@gmail.com>
 */
class InscriptionCheckCommand extends ContainerAwareCommand
{

    /**
     * Configuração do command.
     */
    protected function configure()
    {
        $this
            ->setName('enlab:inscription:check')
            ->setDescription('Job responsible for check inscription status.')
        ;
    }

    /**
     * Executa o command.
     * @param \Symfony\Component\Console\Input\InputInterface $input
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        // Finaliza lotes que já ultrapassaram a data
        $this->checkInscriptions($output);
    }

    /**
     * Verifica inscriptions que precisam ser cancelados.
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function checkInscriptions(OutputInterface $output)
    {
        $output->writeln(sprintf('<comment>[%s]</comment> Canceling inscriptions that has overpassed <info>15 minutes</info>.', date('H:i:s')));

        $em = $this->getContainer()->get('doctrine')->getManager();
        $inscriptionRepository = $this->getInscriptionRepository();
        $inscriptionsToCancel = $inscriptionRepository->findOpenedInscriptions();

        $inscriptionStatusCanceled = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::CANCELADO);
        $paymentStatusCanceled = $em->getReference('AppBundle:PaymentStatus', PaymentStatus::CANCELADA);

        $count = 0;
        foreach ($inscriptionsToCancel as $inscription) {
            // Cancela a matricula/compra
            $inscription->setInscriptionStatus($inscriptionStatusCanceled);
            $em->persist($inscription);

            foreach ($inscription->getPayments() as $payment) {
                // Cancela os pagamento
                $payment->setPaymentStatus($paymentStatusCanceled);
                $em->persist($payment);
            }
            $message = $this->createMessage($inscription);
            $this->sendCancelEmail($inscription->getUser()->getEmail(), "Matrícula Cancelada", $message);
            $count++;
        }
        $output->writeln(sprintf('<comment>%s</comment> inscriptions finished.', $count));
        $em->flush();
    }

    /**
     * Retorna o repositório de Inscription.
     * @return \AppBundle\Repository\InscriptionRepository
     */
    protected function getInscriptionRepository()
    {
        return $this->getContainer()->get('doctrine')->getRepository('AppBundle:Inscription');
    }

    private function createMessage($entity)
    {
        $message = '<div style="width: 100%; background: #004c83; padding: 10px 0;">';
        $message .= '<img src="http://enlab.com.br/bundles/app/img/frontend/logo.png" alt="Enlab" style="margin: 0 auto;display:block;"/>';
        $message .= '</div>';
        $message .= '<div style="width: 100%; background: #f3f3f3; padding: 30px 0; text-align:center;">';
        $message .= '    <h1 style="color: #191919; font-weight:500; font-size: 24px;margin: 20px 0;;">Matr&iacute;cula Cancelada</h1>';
        $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Ol&aacute;, seu pagamento n&atilde;o foi aprovado e por isto a sua matr&iacute;cula foi cancelada</p>';
        $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Voc&ecirc; pode realizar uma nova matr&iacute;cula, clicando no bot&atilde;o abaixo.:</p>';
        $message .= '    <p><a href="http://www.enlab.com.br/curso/'.$entity->getCourse()->getSlug().'"><i style="display: inline-block; padding: 6px 12px; margin-bottom: 0;text-align: center; white-space: nowrap; vertical-align: middle;cursor: pointer;font-style:normal;border-radius: 4px;background: #004c83; border: 1px solid #004c83; color: #fff; font-size: 16px; font-weight: 700; transition: 0.3s; width: 145px; height: 40px; line-height: 2.668571;cursor: pointer">NOVA MATR&Iacute;CULA</i></a>';
        $message .= '</div>';
        return $message;
    }

    /**
     * Enviar email de cancelamento
     * @param String $email
     * @param String $subject
     * @param String $message
     */
    public function sendCancelEmail($email, $subject, $message)
    {
        $mailMessage = \Swift_Message::newInstance()
              ->setSubject("[Enlab] - ".$subject)
              ->setFrom("no-reply@enlab.com.br")
              ->setTo($email)
              ->setContentType("text/html")
              ->setBody($message);

        $this->getContainer()->get('mailer')->send($mailMessage);
    }
}
