<?php

namespace AppBundle\Command;

use AppBundle\Entity\InscriptionStatus;
use AppBundle\Entity\PaymentStatus;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

/**
 * Command responsável pela verificação dos status dos pedidos que precisam ser cancelados.
 */
class WaitingListCourseCommand extends ContainerAwareCommand
{

    /**
     * Configuração do command.
     */
    protected function configure()
    {
        $this->setName('unimed:inscription:waiting-list-course')
             ->setDescription('');
    }

    /**
     * Executa o command.
     * @param \Symfony\Component\Console\Input\InputInterface $input
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        // Finaliza lotes que já ultrapassaram a data
        $this->checkInscriptionsWaitingList($output);
    }

    /**
     * Verifica inscriptions que precisam ser cancelados.
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function checkInscriptionsWaitingList(OutputInterface $output)
    {
        // $output->writeln(sprintf('<comment>[%s]</comment> Canceling inscriptions that has overpassed <info>15 minutes</info>.', date('H:i:s')));

        $em = $this->getContainer()->get('doctrine')->getManager();
        $inscriptionRepository = $this->getInscriptionRepository();
        $courseRepository = $this->getCourseRepository();
        $router = $this->getContainer()->get('router');
        $referenceType = UrlGeneratorInterface::ABSOLUTE_PATH;

        $courses = $courseRepository->findByModality('Presencial');

        $inscriptionStatusWaiting = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::AGUARDANDO_PAGAMENTO);
        $inscriptionStatusApproved = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO);

        foreach ($courses as $index => $course) {
            $inscriptionsWaiting = $inscriptionRepository->getInscriptionsByCoursePresencial($course, $inscriptionStatusWaiting);
            $inscriptionsApproved = $inscriptionRepository->getInscriptionsByCoursePresencial($course, $inscriptionStatusApproved);

            if (count($inscriptionsWaiting) > 0) {
                $quantAvaiableForInscription = $course->getQuantVagas() - count($inscriptionsApproved);
                if ($quantAvaiableForInscription > 0) {
                    for ($i=0; $i < $quantAvaiableForInscription; $i++) {
                        $inscription = $inscriptionsWaiting[$i];
                        $user = $inscription->getUser();
                        $inscription->setInscriptionStatus($inscriptionStatusApproved);
                        $urlCourse = $router->generate('course_modality_show', array('slug' => $course->getSlug()), $referenceType);

                        // Notificar por email da inscricao feita
                        $subject = 'Matricula Curso ' . $course->getName();
                        $message = "Olá ". $user->getFirstName(). ",  <br />";
                        $message .= "você estava na fila de espera para o curso ". $course->getName() ." .";
                        $message .= "Ocorreu uma desistência e você foi incluído em nossa lista de participantes.";
                        $message .= "Para mais informações, acesse http://lmssambatech.com.br". $urlCourse;

                        $email = $user->getEmail();

                        if (!empty($inscription->getEmail())) {
                            $email = $inscription->getEmail();
                        }

                        $this->notificationUserEmail($email, $subject, $message);
                    }
                }
            }

            // $output->writeln('Inscriçoes Ativas - '.count($inscriptionsApproved));
            // $output->writeln('Inscriçoes Aguardando - '.count($inscriptionsWaiting));
        }

        // $output->writeln('Numero de Cursos Presenciais '.count($courses));
        
        // $output->writeln(sprintf('<comment>%s</comment> inscriptions finished.', $count));
        $em->flush();
    }

    /**
     * Retorna o repositório de Inscription.
     * @return \AppBundle\Repository\InscriptionRepository
     */
    protected function getInscriptionRepository()
    {
        return $this->getContainer()->get('doctrine')->getRepository('AppBundle:Inscription');
    }

    /**
     * Retorna o repositório de Inscription.
     * @return \AppBundle\Repository\InscriptionRepository
     */
    protected function getCourseRepository()
    {
        return $this->getContainer()->get('doctrine')->getRepository('AppBundle:Course');
    }

    // private function createMessage($course)
    // {
    //     $message = '<div style="width: 100%; background: #004c83; padding: 10px 0;">';
    //     $message .= '<img src="http://enlab.com.br/bundles/app/img/frontend/logo.png" alt="Enlab" style="margin: 0 auto;display:block;"/>';
    //     $message .= '</div>';
    //     $message .= '<div style="width: 100%; background: #f3f3f3; padding: 30px 0; text-align:center;">';
    //     $message .= '    <h1 style="color: #191919; font-weight:500; font-size: 24px;margin: 20px 0;;">Matr&iacute;cula Realizada</h1>';
    //     $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Ol&aacute;, seu pagamento n&atilde;o foi aprovado e por isto a sua matr&iacute;cula foi cancelada</p>';
    //     $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Voc&ecirc; pode realizar uma nova matr&iacute;cula, clicando no bot&atilde;o abaixo.:</p>';
    //     $message .= '    <p><a href="http://www.enlab.com.br/curso/'.$course->getCourse()->getSlug().'"><i style="display: inline-block; padding: 6px 12px; margin-bottom: 0;text-align: center; white-space: nowrap; vertical-align: middle;cursor: pointer;font-style:normal;border-radius: 4px;background: #004c83; border: 1px solid #004c83; color: #fff; font-size: 16px; font-weight: 700; transition: 0.3s; width: 145px; height: 40px; line-height: 2.668571;cursor: pointer">NOVA MATR&Iacute;CULA</i></a>';
    //     $message .= '</div>';
    //     return $message;
    // }

    /**
     * Enviar email de cancelamento
     * @param String $email
     * @param String $subject
     * @param String $message
     */
    public function notificationUserEmail($email, $subject, $message)
    {
        $mailMessage = \Swift_Message::newInstance()
              ->setSubject("[Unimed] - ".$subject)
              ->setFrom("no-reply@lmssambatech.com.br")
              ->setTo($email)
              ->setContentType("text/html")
              ->setBody($message);

        $this->getContainer()->get('mailer')->send($mailMessage);
    }
}
