<?php

namespace AppBundle\Controller\Backend;

use AppBundle\Entity\AccessControl;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;
use AppBundle\Form\AccessControlType;

/**
 * Accesscontrol controller.
 *
 * @Route("/backend/perfil-de-acesso")
 */
class AccessControlController extends BaseController
{
    /**
     * Lists all AccessControl entities.
     *
     * @Route("/", name="backend_access_control_index")
     * @Template("AppBundle:Backend\AccessControl:index.html.twig")
     * @Method("GET")
     */
    public function indexAction()
    {
        $this->verificationUserAccessControl('index_access_control');

        $em = $this->getDoctrine()->getManager();

        $accessControls = $em->getRepository('AppBundle:AccessControl')->findAll();

        return array(
            'accessControls' => $accessControls,
        );
    }

    /**
     * Creates a new accessControl entity.
     *
     * @Route("/new", name="backend_access_control_new")
     * @Template("AppBundle:Backend\AccessControl:new.html.twig")
     * @Method({"GET", "POST"})
     */
    public function newAction(Request $request)
    {
        $this->verificationUserAccessControl('new_access_control');

        $accessControl = new Accesscontrol();
        $form = $this->createForm('AppBundle\Form\AccessControlType', $accessControl);
        $form->handleRequest($request);

        if ($form->isValid()) {

            $em = $this->getDoctrine()->getManager();
            $em->persist($accessControl);
            $em->flush();

            return $this->redirectToRoute('backend_access_control_index', array());
        }

        return array(
            'accessControl' => $accessControl,
            'form' => $form->createView(),
        );
    }

    /**
     * Displays a form to edit an existing accessControl entity.
     *
     * @Route("/{id}/edit", name="backend_access_control_edit")
     * @Template("AppBundle:Backend\AccessControl:edit.html.twig")
     * @Method({"GET", "POST"})
     */
    public function editAction(Request $request, AccessControl $accessControl)
    {
        $this->verificationUserAccessControl('edit_access_control');

        $deleteForm = $this->createDeleteForm($accessControl);
        $editForm = $this->editCreateForm($accessControl);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('backend_access_control_index', array());
        }

        return array(
            'accessControl' => $accessControl,
            'edit_form' => $editForm->createView(),
            'delete_form' => $deleteForm->createView(),
        );
    }

    /**
     * Deletes a Category entity.
     * @Route("/{id}/delete", name="backend_access_control_delete")
     */
    public function deleteAction(Request $request, $id)
    {
        try {
            $this->verificationUserAccessControl('remove_access_control');

            $em = $this->getDoctrine()->getManager();
            $accessControl = $em->getRepository('AppBundle:AccessControl')->find($id);

            if (!$accessControl) {
                throw $this->createNotFoundException('Unable to find AccessControl entity.');
            }

            $user = $this->getUser();

            $userAccessControl = $em->getRepository('AppBundle:UserAccessControl')->findBy(array(
                'accessControl' => $accessControl,
            ));

            if ($userAccessControl) {
                $msgError = 'Não é possível remover o perfil selecionado, ';
                $msgError .= 'ele está associado a um usuário ativo no sistema.';
                $this->errorMessage($request, $msgError);

                return $this->redirectToRoute('backend_access_control_index');
            }

            $em->remove($accessControl);
            $em->flush();
            $this->successMessage($request, 'Registro excluído com sucesso!');

            return $this->redirect($this->generateUrl('backend_access_control_index'));
        } catch (Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }
    }

    /**
     * Creates a form to create a Category entity.
     *
     * @param AccessControl $accessControl The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function editCreateForm(AccessControl $accessControl)
    {
        $form = $this->createForm(new AccessControlType(), $accessControl, array(
            'action' => $this->generateUrl('backend_access_control_edit', array('id' => $accessControl->getId())),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }

    /**
     * Creates a form to delete a accessControl entity.
     *
     * @param AccessControl $accessControl The accessControl entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createDeleteForm(AccessControl $accessControl)
    {
        return $this->createFormBuilder()
            ->setAction($this->generateUrl('backend_access_control_delete', array('id' => $accessControl->getId())))
            ->setMethod('DELETE')
            ->getForm()
        ;
    }
}
