<?php

namespace AppBundle\Controller\Backend;

use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;
use AppBundle\Entity\CourseClass;
use AppBundle\Entity\CourseClassStatus;
use AppBundle\Entity\InscriptionClassStatus;
use AppBundle\Entity\InscriptionClass;
use AppBundle\Form\CourseClassType;
use AppBundle\Entity\InscriptionImport;
use AppBundle\Form\InscriptionImportType;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * CourseClass controller.
 *
 * @Route("/backend/course-class")
 */
class CourseClassController extends BaseController
{
    /**
     * @Route("/" , name="backend_courseclass")
     * @Template("AppBundle:Backend\CourseClass:index.html.twig")
     */
    public function indexAction()
    {
        $this->verificationUserAccessControl('index_courseclass');
        
        $em = $this->getDoctrine()->getManager();

        $entities = $em->getRepository('AppBundle:CourseClass')->findAll();

        return array(
            'entities' => $entities,
        );
    }

    /**
     * Creates a new CourseClass entity.
     *
     * @Route("/create", name="backend_courseclass_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\CourseClass:new.html.twig")
     */
    public function createAction(Request $request)
    {
        $this->verificationUserAccessControl('new_courseclass');

        $entity = new CourseClass();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);
        // if ($form->isValid()) {s
            if ($this->validDescriptionName($entity)) {
                try {
                    $em = $this->getDoctrine()->getManager();
                    $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::ABERTO);
                    $entity->setCourseClassStatus($courseClassStatus);
                    $this->requestCustomForm($request, $entity);
                    $em->persist($entity);
                    foreach ($entity->getEventClass() as $eventClass) {
                        $eventClass->setCourseClass($entity);
                        $em->persist($eventClass);
                    }
                    $em->flush();

                    $request->getSession()
                        ->getFlashBag()
                        ->add('success', 'Registro criado com sucesso!');

                    return $this->redirect($this->generateUrl('backend_courseclass', array('id' => $entity->getId())));
                } catch (\Exception $e) {
                    $this->log($e);
                    $request->getSession()
                        ->getFlashBag()
                        ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
                }
            }else{
                $request->getSession()->getFlashBag()->add('error', 'A descrição '.$entity->getDescription().' já está sendo utilizado.');
            }

        // }

        return array(
            'entity' => $entity,
            'form' => $form->createView(),
        );
    }

    /**
     * Creates a form to create a CourseClass entity.
     *
     * @param CourseClass $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(CourseClass $entity)
    {
        $form = $this->createForm(new CourseClassType(), $entity, array(
            'action' => $this->generateUrl('backend_courseclass_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to create a new CourseClass entity.
     *
     * @Route("/new", name="backend_courseclass_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction()
    {
        $this->verificationUserAccessControl('new_courseclass');

        $entity = new CourseClass();
        $form = $this->createCreateForm($entity);

        return array(
            'entity' => $entity,
            'form' => $form->createView(),
        );
    }

    /**
     * Displays a form to edit an existing CourseClass entity.
     *
     * @Route("/{id}/edit", name="backend_courseclass_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl('edit_courseclass');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CourseClass entity.');
        }

        $editForm = $this->createEditForm($entity);
        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a CourseClass entity.
    *
    * @param CourseClass $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(CourseClass $entity)
    {
        $form = $this->createForm(new CourseClassType(), $entity, array(
            'action' => $this->generateUrl('backend_courseclass_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing CourseClass entity.
     *
     * @Route("/{id}", name="backend_courseclass_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\CourseClass:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('edit_courseclass');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CourseClass entity.');
        }

        //Variáveis de controle para manter registros a serem excluídos
        $originalEventClass = array();

        // Cria um array com questionItems já existentes
        foreach ($entity->getEventClass() as $eventClass) {
            $originalEventClass[] = $eventClass;
        }

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        // if ($editForm->isValid()) {
            if ($this->validDescriptionName($entity)) {
                try {
                    // Set entity fields custom
                    $this->requestCustomForm($request, $entity);
                    //Tratamento para a persistência de serviços
                    // Filtra o array $originalQuestionItemsAction para conter apenas itens que não existam mais
                    foreach ($entity->getEventClass() as $eventClass) {
                        $eventClass->setCourseClass($entity);
                        foreach ($originalEventClass as $key => $toDel) {
                            if ($toDel->getId() === $eventClass->getId()) {
                                unset($originalEventClass[$key]);
                            }
                        }
                    }

                    // Remove os dias que foram apagadas
                    foreach ($originalEventClass as $eventClass) {
                        $em->remove($eventClass);
                    }
                    $em->flush();

                    $request->getSession()
                        ->getFlashBag()
                        ->add('success', 'Registro atualizado com sucesso!');

                    return $this->redirect($this->generateUrl('backend_courseclass', array('id' => $id)));
                } catch (\Exception $e) {
                    $this->log($e);
                    $request->getSession()->getFlashBag()->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
                }
            }else{
                $request->getSession()->getFlashBag()->add('error', 'A descrição '.$entity->getDescription().' já está sendo utilizado.');
            }
        // }

        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }

    /**
     * Deletes a CourseClass entity.
     * @Route("/{id}/delete", name="backend_courseclass_delete")
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_courseclass');

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CourseClass entity.');
        }

        if(count($entity->getInscriptionsClass()) > 0){
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Esta turma/evento não pode ser excluído pois existe matrículas associadas!');
            return $this->redirect($this->generateUrl('backend_courseclass'));
        }
        $em->remove($entity);
        $em->flush();
        $request->getSession()
            ->getFlashBag()
            ->add('success', 'Registro excluído com sucesso!');
        return $this->redirect($this->generateUrl('backend_courseclass'));
    }

    /**
     * Set entity fields custom
     * @param Request $request
     * @param Entity $entity
     */
    public function requestCustomForm($request, $entity)
    {
        $em = $this->getDoctrine()->getManager();
        $customArray = $request->request->get('custom');

        if (!empty($customArray['dtBegin'])) {
            $dtBegin = \DateTime::createFromFormat('d/m/Y H:i', $customArray['dtBegin']);
            $entity->setDtBegin($dtBegin);
        }

        if (!empty($customArray['dtEnd'])) {
            $dtEnd = \DateTime::createFromFormat('d/m/Y H:i', $customArray['dtEnd']);
            $entity->setDtEnd($dtEnd);
        }
    }


    /**
     * Valida se o username e email são válidos
     * @param User $entity
     * @return boolean $flag
     */
    public function validDescriptionName($entity)
    {
        $repository = $this->getCourseClassRepository();
        $flag = $repository->validateCourseClassDescription($entity);

        return $flag;
    }

    /**
     * Lists all Course entities.
     *
     * @Route("/export/{id}/exportacao", name="backend_course_class_export")
     * @Method("GET")
     * @Template()
     */
    public function indexExportAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("execute_script_course");
        set_time_limit(0);
        ini_set('mysql.connect_timeout', '0');
        ini_set('max_execution_time', '0');
        $em = $this->getDoctrine()->getManager();
        //create the prepared statement, by getting the doctrine connection
        $script = "select c.title as 'Curso', cc.description as 'Turma', u.id as 'User id', concat(concat(u.first_name,' '), u.last_name) as 'Nome', ic.grade as 'Nota', ic.observation as 'Observaçao', ics.name as 'Status', DATE_FORMAT(ic.dt_creation, '%d/%m/%Y %H:%i:%s') as 'Data De Criação', DATE_FORMAT(ic.dt_creation, '%d/%m/%Y %H:%i:%s') as 'Data De Atualização'
        from inscription_class ic
        join course_class cc on cc.id = ic.course_class_id
        join course c on c.id = cc.course_id
        left join inscriptionclass_status ics on ics.id = ic.inscriptionclass_status_id
        join user u on u.id = ic.user_id
        where ic.course_class_id = ".$id;
        $stmt = $em->getConnection()->prepare($script);
        $stmt->execute();

        $list = $stmt->fetchAll();
        $rows = array();
        if (count($list)>0) {
            $headers = array_keys($list[0]);
            foreach ($list as $key => $value) {
                $data = array();
                foreach ($headers as $index => $header) {
                    // $cabecalho = $header[$index];
                    $data[] = $value[$header];
                }
                $rows[] = implode(',', $data);
            }
            $contentHeader = implode(',', $headers);
            $contentBody = implode("\n", $rows);
            $content = $contentHeader ."\n".$contentBody;
            $response = new Response($content);
            $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');

            return $response;
        } else {
            $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Não foram encontrados dados para exportação!');

            return $this->redirect($this->generateUrl('backend_course_class'));
        }
    }

    /**
     * Lists all learnings entities.
     *
     * @Route("/{id}/learnings", name="backend_course_class_learnings")
     * @Method({"GET", "POST"})
     * @Template()
     */
    public function learningsAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_class_all");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        if ($request->isMethod('POST')) {
            $verify = 0;
            $inscriptions = $request->request->get("inscription");
            foreach ($inscriptions as $key => $inscription) {
                $inscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->find($key);
                $status = $em->getRepository('AppBundle:InscriptionClassStatus')->find($inscription['status']);
                // if($inscription['grade']){
                //     $inscriptionClass->setGrade(intval($inscription['grade']));
                // }
                $inscriptionClass->setObservation($inscription['observation']);
                $inscriptionClass->setInscriptionClassStatus($status);
                $em->persist($inscriptionClass);
                $verify = 1;
            }

            if ($verify == 1) {

                $toFinish = true;
                foreach ($entity->getInscriptionsClass() as $inscriptionsClass) {
                    if(!$inscriptionsClass->getInscriptionClassStatus()){
                        $toFinish = false;
                    }
                }

                if($toFinish){
                    $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::FECHADO);
                    $entity->setCourseClassStatus($courseClassStatus);
                    $em->persist($entity);
                }else{
                    $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::ABERTO);
                    $entity->setCourseClassStatus($courseClassStatus);
                    $em->persist($entity);
                }
                $em->flush();

                $request->getSession()
                    ->getFlashBag()
                    ->add('success', 'Matrículas atualizadas com sucesso!');
            }
        }

        $inscriptions = $this->getInscriptionClassRepository()->findByCourseClass($entity);


        return array(
            'entity'  => $entity,
            'inscriptions' => $inscriptions,
            'id' => $id
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/change-status-mass", name="backend_course_class_inscriptions_status")
     * @Method("POST")
     * @Template("AppBundle:Backend\CourseClass:learnings.html.twig")
     */
    public function inscriptionMassChangeStatusAction(Request $request, $id)
    {
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);
        $inscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->findByCourseClass($courseClass);
        $statusId = $request->request->get("status");
        $verify = 0;
        $status = $em->getRepository('AppBundle:InscriptionClassStatus')->find($statusId);
        foreach ($inscriptionClass as $item) {        
            $item->setInscriptionClassStatus($status);
            $em->persist($item);
            $verify = 1;
        }
        if ($verify == 1) {
            $toFinish = true;
            foreach ($courseClass->getInscriptionsClass() as $inscriptionsClass) {
                if(!$inscriptionsClass->getInscriptionClassStatus()){
                    $toFinish = false;
                }
            }
            
            if($toFinish){
                $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::FECHADO);
                $courseClass->setCourseClassStatus($courseClassStatus);
                $em->persist($courseClass);
            }else{
                $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::ABERTO);
                $courseClass->setCourseClassStatus($courseClassStatus);
                $em->persist($courseClass);
            }

            $em->flush();
            $request->getSession()
                ->getFlashBag()
                ->add('success', 'Status alterado com sucesso!');
        }

        return $this->redirectToRoute('backend_course_class_learnings', array(
            'id' => $id,
        ));
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription", name="backend_course_class_inscriptions")
     * @Method("GET")
     * @Template()
     */
    public function inscriptionsAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_class_all");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'entity'  => $entity,
            'id' => $id
        );
    }

    public function getCountInscriptionFromClassId($classId)
    {
        $repositoryInscriptionClass = $this->getInscriptionClassRepository();
        $countInscriptions = $repositoryInscriptionClass->findTotalInscriptions($classId);

        return $countInscriptions;
    }

    public function getCountClients()
    {
        $clientRepository = $this->getClientRepository();
        $countClients = $clientRepository->findTotalClients(true);

        return $countClients;
    }

    /**
     * @Route("/inscription/{courseClassId}/menu-options", name="course_class_menu_options")
     */
    public function inscriptionMenuOptionAction($courseClassId)
    {
        $courseClass = $this->getCourseClassRepository()->find($courseClassId);
        $userGroups = $this->getUserGroupRepository()->findAll();
        return $this->render('AppBundle:Backend\CourseClass:menu_options_index.html.twig', array(
            'courseClass' => $courseClass,
            'userGroups' => $userGroups
        ));
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/setor", name="backend_course_class_inscriptions_sector")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsSector.html.twig")
     */
    public function inscriptionsSectorAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();
        $query = $this->getRequest()->get("query");
        
        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);
        
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find CourseClass entity.');
        }
        
        $sector = $em->getRepository('AppBundle:Sector')->findForSearch($query);

        //paginator sector
        $paginatorsector  = $this->get('knp_paginator');
        $paginationsector = $paginatorsector->paginate($sector, $request->query->get('page', 1), 10);


        return array(
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'sector' => $paginationsector,
            'entity'  => $entity,
            'query' => $query
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/cargos", name="backend_course_class_inscriptions_cargos")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsJobRole.html.twig")
     */
    public function inscriptionsCargoAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();
        $query = $this->getRequest()->get("query");

        $entity = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $jobrole = $em->getRepository('AppBundle:JobRole')->findForSearch($query);

        //paginator jobrole
        $paginatorjobrole  = $this->get('knp_paginator');
        $paginationjobrole = $paginatorjobrole->paginate($jobrole, $request->query->get('page', 1), 10);

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'jobrole' => $paginationjobrole,
            'entity'  => $entity,
            'query' => $query
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/group-user/{group_user}", name="backend_course_class_inscriptions_group_user")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsGroupUser.html.twig")
     */
    public function inscriptionByGroupUserAction(Request $request, $id, $group_user)
    {
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);
        $clientService = $this->container->get('client.service');
        $query = $this->getRequest()->get("query");

        if (!$courseClass) {
            throw $this->createNotFoundException('Curso não encontrado.');
        }
        
        $userGroup = $this->getUserGroupRepository()->find($group_user);
        $clients = $this->getClientRepository()->findByPagination($query, $userGroup, true);

        $paginator = $this->get('knp_paginator');
        $paginator = $paginator->paginate($clients, $request->query->get('page', 1), 10);

        return array(
            'courseClass' => $courseClass,
            'clients' => $paginator,
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'group_user' => $group_user,
            'query' => $query,
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/matricula", name="backend_course_class_inscriptions_matricula")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsClients.html.twig")
     */
    public function inscriptionsRegistrationction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_client");
        $em = $this->getDoctrine()->getManager();
        $query = $this->getRequest()->get("query");

        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$courseClass) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $clients = $em->getRepository('AppBundle:Client')->findByPagination($query, null, true);

        //paginator client
        $paginatorclient  = $this->get('knp_paginator');
        $paginationclient = $paginatorclient->paginate($clients, $request->query->get('page', 1), 10);


        return array(
            'clients' => $paginationclient,
            'courseClass'  => $courseClass,
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'query' => $query
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/all", name="backend_course_class_inscriptions_all")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsAll.html.twig")
     */
    public function inscriptionsAllAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_active");
        $em = $this->getDoctrine()->getManager();

        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);

        if (!$courseClass) {
            throw $this->createNotFoundException('Unable to find CourseClass entity.');
        }

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'courseClass'  => $courseClass,
        );
    }


    /**
     * Realize import for inscriptions.
     *
     * @Route("/{id}/inscription/import", name="backend_course_class_inscriptions_import")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsImport.html.twig")
     */
    public function inscriptionsImportAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_active");
        $em = $this->getDoctrine()->getManager();
        
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);
        
        if (!$courseClass) {
            throw $this->createNotFoundException('Unable to find CourseClass entity.');
        }
        $inscriptionImport = new InscriptionImport();
        $form = $this->createForm(new InscriptionImportType(), $inscriptionImport);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->log('[inscriptionsImportAction] PROCESSANDO O ARQUIVO');

            $em = $this->getDoctrine()->getManager();

            $inputFileName = $inscriptionImport->getFile();

            try {
                $inputFileType = \PHPExcel_IOFactory::identify($inputFileName);
                $objReader = \PHPExcel_IOFactory::createReader($inputFileType);
                $objPHPExcel = $objReader->load($inputFileName);
            } catch (\Exception $e) {
                // die('Error loading file "' . pathinfo($inputFileName, PATHINFO_BASENAME) . '": ' . $e->getMessage());
                $this->log($e);
                $this->errorMessage($request, 'Ocorreu um erro ao processar o arquivo. Verifique o padrão utilizado e tente novamente.');
            }

            //  Get worksheet dimensions
            $sheet = $objPHPExcel->getSheet(0);
            $highestRow = $sheet->getHighestRow();
            $highestColumn = $sheet->getHighestColumn();

            $total = 0;

            try {
                $this->log('[inscriptionsImportAction] ##################');
                $this->log('[inscriptionsImportAction] Loop nas linhas do arquivo');
                $inscriptionService = $this->container->get('inscription.service');
                //  Loop through each row of the worksheet in turn
                for ($row = 2; $row <= $highestRow; $row++) {
                    //  Read a row of data into an array
                    $rowData = $sheet->rangeToArray('A' . $row . ':' . $highestColumn . $row, null, true, false);
                    //Obtem o registro da linha
                    $userRow = $rowData[0];

                    if(($userRow[0] && $userRow[1] && $userRow[2]) || $userRow[3]){
                        $userItem = $em->getRepository('AppBundle:Client')->findOneBy(
                            array(
                                "mat_colaborador" => $userRow[0],
                                "num_enterprise" => $userRow[1],
                                "tip_colaborador" => $userRow[2]
                            )
                        );
                        if(!$userItem){
                            $userItem = $em->getRepository('AppBundle:Client')->findOneBy(
                                array(
                                    "login" => $userRow[3]
                                )
                            );
                        }

                        if(!$userItem){
                            $this->errorMessage($request, "<b>Registro da linha ".$row." não encontrado.</b> <br>");
                        }else{
                            if($userItem->isEnabled()){
                                $userInscriptionAnotherClass = $em->getRepository('AppBundle:InscriptionClass')->findAnotherInscription($userItem, $courseClass);
                                if(!$userInscriptionAnotherClass){
                                    $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($userItem, $courseClass);
                                    if ($userInscriptionClass == null) {
                                        $inscriptionClass = new InscriptionClass();
                                        $inscriptionClass->setUser($userItem);
                                        $inscriptionClass->setCourseClass($courseClass);
                                        $em->persist($inscriptionClass);
                                        $em->flush();
                                        //Verifica o status da turma
                                        $this->statusCourseClass($em, $courseClass);
                                    }
                                }else{
                                    $this->errorMessage($request, "<b>".$userItem->getFullName()."</b> da <b>linha ".$row."</b> já está matriculado na turma <b>".$userInscriptionAnotherClass->getCourseClass()->getDescription()."</b> no mesmo período de data.<br>");
                                }
                            }else{
                                $this->errorMessage($request, "<b>".$userItem->getFullName()."</b> da <b>linha ".$row."</b> está inativo.<br>");
                            }
                        }
                    }
                    $em->flush();
                    $this->log($userRow);
                }
            } catch (\Exception $e) {
                $this->log("ERRO: ");
                $this->log($e);
                $this->get('session')->getFlashBag()->add('error', "Ocorreu um erro ao realizar a importacão.");
                // die('Error loading file "' . pathinfo($inputFileName, PATHINFO_BASENAME) . '": ' . $e->getMessage());
            }

            $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Importação de matrículas realizada com sucesso.');
        }

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromClassId($id),
            'count_clients' => $this->getCountClients(),
            'courseClass'  => $courseClass,
            'form' => $form->createView(),
        );
    }

    /**
     * Activate a Inscription entity.
     *
     * @Route("/class/{courseClassId}/activate-sector/inscription/{sector_id}/sector", name="backend_inscription_class_activate_sector")
     *
     */
    public function activateInscriptionSectorAction(Request $request, $courseClassId, $sector_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($courseClassId);
        $sector = $em->getRepository('AppBundle:Sector')->find($sector_id);
        $user = $em->getRepository('UserBundle:User')->findBy(array('sector' => $sector,'enabled' => 1));

        if (!$courseClass) {
            throw $this->createNotFoundException('Registro não encontrado de Turma.');
        }

        if (!$user) {
            $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Nenhum funcionário cadastrado nesse setor');
            return $this->redirect($this->generateUrl('backend_course_class_inscriptions_sector', array("id" => $courseClass->getId())));
        }

        try {
            $verify = 0;
            $enrolled = 0;
            $msg = "";
            foreach ($user as $us) {
                $userInscriptionAnotherClass = $em->getRepository('AppBundle:InscriptionClass')->findAnotherInscription($us, $courseClass);
                if(!$userInscriptionAnotherClass){
                    $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($us, $courseClass);
                    if ($userInscriptionClass == null) {
                        $inscriptionClass = new InscriptionClass();
                        $inscriptionClass->setUser($us);
                        $inscriptionClass->setCourseClass($courseClass);
                        $em->persist($inscriptionClass);
                        $verify = 1;
                    }
                }else{
                    $enrolled = 1;
                    $msg .="<b>".$us->getFullName()."</b> já está matriculado na turma <b>".$userInscriptionAnotherClass->getCourseClass()->getDescription()."</b> no mesmo período de data.<br>";
                }
            }
            if ($verify == 1) {
                $em->flush();
                //Verifica o status da turma
                $this->statusCourseClass($em, $courseClass);
                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro liberado com sucesso!');
            } else {
                if($enrolled == 1){
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', $msg);
                }else{
                    $request->getSession()
                       ->getFlashBag()
                       ->add('error', 'Nenhum funcionário desse setor para ser cadastrado');
                }
            }
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_class_inscriptions_sector', array("id" => $courseClass->getId())));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/class/{courseClassId}/activate-jobrole/inscription/{jobrole_id}/jobrole", name="backend_inscription_class_activate_jobrole")
     *
     */
    public function activateInscriptionJobRoleAction(Request $request, $courseClassId, $jobrole_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($courseClassId);
        $job = $em->getRepository('AppBundle:JobRole')->find($jobrole_id);
        $user = $em->getRepository('UserBundle:User')->findBy(array('jobRole' => $job,'enabled' => 1));

        if (!$courseClass) {
            throw $this->createNotFoundException('Registro não encontrado de Turma.');
        }

        if (!$user) {
            $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Nenhum funcionário cadastrado nesse cargo');
            return $this->redirect($this->generateUrl('backend_course_class_inscriptions_cargos', array("id" => $courseClass->getId())));
        }

        try {
            $verify = 0;
            $enrolled = 0;
            $msg = "";
            foreach ($user as $us) {
                $userInscriptionAnotherClass = $em->getRepository('AppBundle:InscriptionClass')->findAnotherInscription($us, $courseClass);
                if(!$userInscriptionAnotherClass){
                    $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($us, $courseClass);
                    if ($userInscriptionClass == null) {
                        $inscriptionClass = new InscriptionClass();
                        $inscriptionClass->setUser($us);
                        $inscriptionClass->setCourseClass($courseClass);
                        $em->persist($inscriptionClass);
                        $verify = 1;
                    }
                }else{
                    $enrolled = 1;
                    $msg .="<b>".$us->getFullName()."</b> já está matriculado na turma <b>".$userInscriptionAnotherClass->getCourseClass()->getDescription()."</b> no mesmo período de data.<br>";
                }
            }
            if ($verify == 1) {
                $em->flush();
                //Verifica o status da turma
                $this->statusCourseClass($em, $courseClass);
                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro liberado com sucesso!');
            } else {
                if($enrolled == 1){
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', $msg);
                }else{
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Nenhum funcionário desse cargo para ser cadastrado');
                }
            }
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_class_inscriptions_cargos', array("id" => $courseClass->getId())));
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/group-user/{group_user}/register", name="backend_course_class_inscriptions_group_user_register")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsGroupUser.html.twig")
     */
    public function inscriptionInMassByGroupUserAction(Request $request, $id, $group_user)
    {
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);
        $query = $this->getRequest()->get("query");

        $userGroup = $this->getUserGroupRepository()->find($group_user);
        $clientsQb = $this->getClientRepository()->findByPagination($query, $userGroup, true);
        $clients = $clientsQb->getResult();
        $verify = 0;
        $enrolled = 0;
        $msg = "";
        foreach ($clients as $client) {
            $userInscriptionAnotherClass = $em->getRepository('AppBundle:InscriptionClass')->findAnotherInscription($client, $courseClass);
            if(!$userInscriptionAnotherClass){
                $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($client, $courseClass);
                if ($userInscriptionClass == null) {
                    $inscriptionClass = new InscriptionClass();
                    $inscriptionClass->setUser($client);
                    $inscriptionClass->setCourseClass($courseClass);
                    $em->persist($inscriptionClass);
                    $verify = 1;
                }
            }else{
                $enrolled = 1;
                $msg .="<b>".$client->getFullName()."</b> já está matriculado na turma <b>".$userInscriptionAnotherClass->getCourseClass()->getDescription()."</b> no mesmo período de data.<br>";
            }
        }
        if ($verify == 1) {
            $em->flush();
            $this->statusCourseClass($em, $courseClass);
            $request->getSession()
                ->getFlashBag()
                ->add('success', 'Matrículas realizadas com sucesso!');
        } else {
            if($enrolled == 1){
                $request->getSession()
                ->getFlashBag()
                ->add('error', $msg);
            }else{
                $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Ninguém foi encontrado para ser matriculado');
            }
        }

        return $this->redirectToRoute('backend_course_class_inscriptions_group_user', array(
            'id' => $id,
            'group_user' => $group_user,
            'query' => $query
        ));
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/group-user/{group_user}/unregister", name="backend_course_inscriptions_group_user_unregister")
     * @Method("GET")
     * @Template("AppBundle:Backend\CourseClass:inscriptionsGroupUser.html.twig")
     */
    public function unscriptionInMassByGroupUserAction(Request $request, $id, $group_user)
    {
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($id);
        $query = $this->getRequest()->get("query");

        $userGroup = $this->getUserGroupRepository()->find($group_user);
        $clientsQb = $this->getClientRepository()->findByPagination($query, $userGroup, true);
        $clients = $clientsQb->getResult();
        $verify = 0;
        foreach ($clients as $client) {
            $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($client, $courseClass);
            if ($userInscriptionClass) {
                $em->remove($userInscriptionClass);
                $verify = 1;
            }
        }
        if ($verify == 1) {
            $em->flush();

            //Verifica o status da turma
            $this->statusCourseClass($em, $courseClass);

            $request->getSession()
                ->getFlashBag()
                ->add('success', 'Desmatrículas realizadas com sucesso!');
        } else {
            $request->getSession()
                ->getFlashBag()
                ->add('error', 'Ninguém foi encontrado para ser desmatriculado');
        }

        return $this->redirectToRoute('backend_course_class_inscriptions_group_user', array(
            'id' => $id,
            'group_user' => $group_user,
            'query' => $query
        ));
    }

    /**
     * Activate Inscription the a User.
     *
     * @Route("/class/{courseClassId}/activate/inscription/{user_id}/{group_user}", name="backend_inscription_class_activate")
     *
     */
    public function activateInscriptionAction(Request $request, $courseClassId, $user_id, $group_user = '')
    {
        $this->verifyAdmin();
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($courseClassId);
        $user = $em->getRepository('UserBundle:User')->find($user_id);
        $inscriptionService = $this->container->get('inscription.service');

        if (!$courseClass) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$user) {
            throw $this->createNotFoundException('Registro não encontrado de Funcionário.');
        }

        try {
            $userInscriptionAnotherClass = $em->getRepository('AppBundle:InscriptionClass')->findAnotherInscription($user, $courseClass);
            if(!$userInscriptionAnotherClass){
                $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($user, $courseClass);
                if ($userInscriptionClass == null) {
                    $inscriptionClass = new InscriptionClass();
                    $inscriptionClass->setUser($user);
                    $inscriptionClass->setCourseClass($courseClass);
                    $em->persist($inscriptionClass);
                    $em->flush();
                    //Verifica o status da turma
                    $this->statusCourseClass($em, $courseClass);
                }
            }else{
                $this->errorMessage($request, "<b>".$user->getFullName()."</b> já está matriculado na turma <b>".$userInscriptionAnotherClass->getCourseClass()->getDescription()."</b> no mesmo período de data.<br>");
            }
        } catch (\Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        if ($group_user) {
            return $this->redirect($this->generateUrl('backend_course_class_inscriptions_group_user', array(
                "id" => $courseClass->getId(),
                "group_user" => $group_user,
            )));
        }

        return $this->redirect($this->generateUrl('backend_course_class_inscriptions_matricula', array(
            "id" => $courseClass->getId()
        )));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/class/{courseClassId}/cancel/inscription/{sector_id}/sector", name="backend_inscription_class_cancel_sector")
     *
     */
    public function cancelInscriptionSectorAction(Request $request, $courseClassId, $sector_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($courseClassId);
        $sector = $em->getRepository('AppBundle:Sector')->find($sector_id);
        if (!$courseClass) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }
        try {
            $users = $em->getRepository('UserBundle:User')->findBySector($sector);  
            foreach ($users as $us) {
                $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($us, $courseClass);
                if($userInscriptionClass){
                    $em->remove($userInscriptionClass);
                }
            }
            $em->flush();
            //Verifica o status da turma
            $this->statusCourseClass($em, $courseClass);
            $request->getSession()
                ->getFlashBag()
                ->add('success', 'Registro cancelado com sucesso!');
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_class_inscriptions_sector', array("id" => $courseClass->getId())));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/class/{courseClassId}/cancel/inscription/{jobrole_id}/jobrole", name="backend_inscription_class_cancel_jobrole")
     *
     */
    public function cancelInscriptionJobRoleAction(Request $request, $courseClassId, $jobrole_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($courseClassId);
        $job = $em->getRepository('AppBundle:JobRole')->find($jobrole_id);       

        try {
            $users = $em->getRepository('UserBundle:User')->findByJobRole($job);
            foreach ($users as $us) {
                $userInscriptionClass = $em->getRepository('AppBundle:InscriptionClass')->getLastInscriptionByUser($us, $courseClass);
                if($userInscriptionClass){
                    $em->remove($userInscriptionClass);
                }
            }
            $em->flush();
            //Verifica o status da turma
            $this->statusCourseClass($em, $courseClass);
            $request->getSession()
                ->getFlashBag()
                ->add('success', 'Registro cancelado com sucesso!');
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_class_inscriptions_cargos', array("id" => $courseClass->getId())));
    }


    /**
     * Cancels a Inscription entity.
     *
     * @Route("/class/{courseClassId}/cancel/inscription/{id}/{group_user}", name="backend_inscription_class_cancel")
     *
     */
    public function cancelInscriptionAction(Request $request, $courseClassId, $id, $group_user = '')
    {
        $this->verificationUserAccessControl("index_inscription_course_all");
        $em = $this->getDoctrine()->getManager();
        $courseClass = $em->getRepository('AppBundle:CourseClass')->find($courseClassId);
        $entity = $em->getRepository('AppBundle:InscriptionClass')->find($id);

        if (!$courseClass) {
            throw $this->createNotFoundException('Registro não encontrado de Turma.');
        }

        if (!$entity) {
            throw $this->createNotFoundException('Registro não encontrado de Matrícula.');
        }

        try {
            $em->remove($entity);
            $em->flush();
            //Verifica o status da turma
            $this->statusCourseClass($em, $courseClass);
            $request->getSession()
               ->getFlashBag()
               ->add('success', 'Registro cancelado com sucesso!');
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        if ($group_user) {
            return $this->redirect($this->generateUrl('backend_course_class_inscriptions_group_user', array(
                "id" => $courseClass->getId(),
                "group_user" => $group_user,
            )));
        }

        return $this->redirect($this->generateUrl('backend_course_class_inscriptions_matricula', array("id" => $courseClass->getId())));
    }

    /**
     *
     * @Route("/class/{courseClassId}/cancel/{user_id}", name="course_class_cancel_subscribe", options={"expose"=true})
     */
    public function cancelInscription(Request $request, $courseClassId, $user_id)
    {
        $em   = $this->getDoctrine()->getManager();
        $user = $em->getRepository('UserBundle:User')->findOneById($user_id);
        $courseClass = $em->getRepository('AppBundle:CourseClass')->findOneById($courseClassId);

        if (!$user) {
            $this->warningMessage($request, 'Usuario não encontrado');
            return new Response(json_encode(array(
                'status' => 500,
                'response' => 'done',
                'msg' => 'Usuario não encontrado'
            )));
        }

        $inscription = $em->getRepository('AppBundle:InscriptionClass')->findOneBy(array(
            'user' => $user,
            'courseClass' => $courseClass,
        ));

        if (!$inscription) {
            $msg = 'Curso não encontrado';
            $status = 500;
        } else {
            $status = 200;
            $msg = 'Matrícula cancelada com sucesso';
            $em->remove($inscription);
            $em->flush();
            //Verifica o status da turma
            $this->statusCourseClass($em, $courseClass);
        }

        return new Response(json_encode(array(
            'status' => $status,
            'response' => 'done',
            'msg' => $msg
        )));
    }

    /**
     * @Route("/class/get-matricula/{userId}", name="course_class_matricula")
     */
    public function getUserMatricula($userId)
    {
        $em = $this->getDoctrine()->getManager();
        $client = $em->getRepository('AppBundle:Client')->findOneById($userId);
        $matricula = " - ";
        if($client){
            $matricula = $client->getMatColaborador();
        }

        return $this->render('AppBundle:Backend\CourseClass:clientMatricula.html.twig', array(
            'matricula' => $matricula,
        ));
    }

    public function statusCourseClass($em, $courseClass){
        $toFinish = true;
        foreach ($courseClass->getInscriptionsClass() as $inscriptionsClass) {
            if(!$inscriptionsClass->getInscriptionClassStatus()){
                $toFinish = false;
            }
        }

        if($toFinish){
            $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::FECHADO);
            $courseClass->setCourseClassStatus($courseClassStatus);
            $em->persist($courseClass);
        }else{
            $courseClassStatus = $em->getReference('AppBundle:CourseClassStatus', CourseClassStatus::ABERTO);
            $courseClass->setCourseClassStatus($courseClassStatus);
            $em->persist($courseClass);
        }
        $em->flush();
    }
}
