<?php

namespace AppBundle\Controller\Backend;

use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Course;
use AppBundle\Entity\CourseAccessControl;
use AppBundle\Entity\Inscription;
use AppBundle\Entity\InscriptionStatus;
use AppBundle\Entity\PaymentStatus;
use AppBundle\Form\CourseType;
use AppBundle\Entity\InscriptionImport;
use AppBundle\Form\InscriptionImportType;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Course controller.
 *
 * @Route("/privado/course")
 */
class CourseController extends BaseController
{

    /**
     * Lists all Course entities.
     *
     * @Route("/", name="backend_course")
     * @Method("GET")
     * @Template()
     */
    public function indexAction()
    {
        $this->verificationUserAccessControl("index_course");
        $em = $this->getDoctrine()->getManager();

        $courses = array();
        $entities = array();
        
        if ($this->get('security.authorization_checker')->isGranted('ROLE_SUPER_ADMIN')) {
            $entities = $em->getRepository('AppBundle:Course')->findAll();
        } else {
            $companies = $this->findCompanyClientResponsible($this->getUser());
            if ($companies) {
                foreach ($companies as $key => $company) {
                    $entities = array_merge($company->getCourses()->toArray(), $entities);
                }
            }
        }

        return array(
            'entities' => $entities,
        );
    }

    /**
     * Lists all Course entities.
     *
     * @Route("/export/{id}/exportacao-{slug}", name="backend_course_export")
     * @Method("GET")
     * @Template()
     */
    public function indexExportAction(Request $request, $id, $slug)
    {
        $this->verificationUserAccessControl("execute_script_course");
        set_time_limit(0);
        ini_set('mysql.connect_timeout', '0');
        ini_set('max_execution_time', '0');
        $em = $this->getDoctrine()->getManager();
        //create the prepared statement, by getting the doctrine connection
        $script = file_get_contents("script.txt");
        $course_placeholder = ["@course_id@"];
        $course_id = [$id];
        $script = str_replace($course_placeholder, $course_id, $script);
        // $inputFileName = dirname(__FILE__).'/../../../web/arquivos/lpe-9501-9929.xlsx';
        $stmt = $em->getConnection()->prepare($script);
        $stmt->execute();

        $list = $stmt->fetchAll();
        $rows = array();
        if (count($list)>0) {
            $headers = array_keys($list[0]);
            foreach ($list as $key => $value) {
                $data = array();
                foreach ($headers as $index => $header) {
                    // $cabecalho = $header[$index];
                    $data[] = $value[$header];
                }
                $rows[] = implode(',', $data);
            }
            $contentHeader = implode(',', $headers);
            $contentBody = implode("\n", $rows);
            $content = $contentHeader ."\n".$contentBody;
            $response = new Response($content);
            $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');

            return $response;
        } else {
            $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Não foram encontrados dados para exportação!');

            return $this->redirect($this->generateUrl('backend_course'));
        }
    }

    /**
     * Creates a new Course entity.
     *
     * @Route("/", name="backend_course_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\Course:new.html.twig")
     */
    public function createAction(Request $request)
    {
        $this->verificationUserAccessControl("new_course");
        $entity = new Course();
        $userGroups = $this->getUserGroupRepository()->findAll();
        $categoryPresencial = $this->getCategoryRepository()->findByModality('Presencial');
        $categoryEad = $this->getCategoryRepository()->findByModality('EAD');
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);

        if ($form->isValid()) {
            try {
                if($entity->getPontuation() && !$entity->getSalutarisId()){
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Para cursos com <b>Pontuação</b> você deve preencher o <b>Id no Salutaris</b>!');
                    return array(
                        'entity' => $entity,
                        'form'   => $form->createView(),
                        'userGroups' => $userGroups,
                    );
                }
                if($entity->getSalutarisId() && !$entity->getPontuation()){
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Para cursos com <b>Id no Salutaris</b> você deve preencher a <b>Pontuação</b>!');
                    return array(
                        'entity' => $entity,
                        'form'   => $form->createView(),
                        'userGroups' => $userGroups,
                    );
                }
                $em = $this->getDoctrine()->getManager();
                $em->persist($entity);
                $em->flush();

                $autoInscriptionOptions = $request->request->get('autoInscription');
                if(!empty($autoInscriptionOptions)){
                    foreach ($autoInscriptionOptions as $key => $value) {
                        $courseAccessControl = new CourseAccessControl();
                        $courseAccessControl->setCourse($entity);
                        $courseAccessControl->setAccess($value);
                        $em->persist($courseAccessControl);
                        $courseAccessControl->setAccess($value);
                    }
                }
                $em->flush();

                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro criado com sucesso!');

                return $this->redirect($this->generateUrl('backend_course', array('id' => $entity->getId())));
            } catch (\Exception $e) {
                $this->log("!!!!");
                $this->log($e);
                $request->getSession()
                ->getFlashBag()
                ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity' => $entity,
            'form'   => $form->createView(),
            'userGroups' => $userGroups,
            'categoryPresencial'=> $categoryPresencial,
            'categoryEad'=> $categoryEad,
        );
    }

    /**
     *
     * @Route("/verify/attempts/{course}/{inscription}", name="backend_course_verify_attempts")
     */
    public function verifyAttemptsAction(Request $request)
    {
        $em = $this->getDoctrine()->getManager();

        return array(
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Creates a form to create a Course entity.
     *
     * @param Course $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Course $entity)
    {
        $this->verificationUserAccessControl("new_course");
        if ($this->get('security.authorization_checker')->isGranted('ROLE_SUPER_ADMIN')) {
            $courseType = new CourseType();
        } else {
            $courseType = new CourseType($this->getUser()->getCompany());
        }
        $form = $this->createForm($courseType, $entity, array(
            'action' => $this->generateUrl('backend_course_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to create a new Course entity.
     *
     * @Route("/new", name="backend_course_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction()
    {
        $this->verificationUserAccessControl("new_course");
        $entity = new Course();
        $userGroups = $this->getUserGroupRepository()->findAll();
        $form   = $this->createCreateForm($entity);
        $categoryPresencial = $this->getCategoryRepository()->findByModality('Presencial');
        $categoryEad = $this->getCategoryRepository()->findByModality('EAD');

        return array(
            'entity' => $entity,
            'userGroups' => $userGroups,
            'form'   => $form->createView(),
            'categoryPresencial'=> $categoryPresencial,
            'categoryEad'=> $categoryEad,
        );
    }

    /**
     * Displays a form to edit an existing Course entity.
     *
     * @Route("/{id}/edit", name="backend_course_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl("edit_course");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Course')->find($id);
        $userGroups = $this->getUserGroupRepository()->findAll();
        $categoryPresencial = $this->getCategoryRepository()->findByModality('Presencial');
        $categoryEad = $this->getCategoryRepository()->findByModality('EAD');

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $editForm = $this->createEditForm($entity);

        return array(
            'entity'      => $entity,
            'userGroups' => $userGroups,
            'edit_form'   => $editForm->createView(),
            'categoryPresencial'=> $categoryPresencial,
            'categoryEad'=> $categoryEad,
        );
    }

    /**
    * Creates a form to edit a Course entity.
    *
    * @param Course $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Course $entity)
    {
        $this->verificationUserAccessControl("edit_course");
        if ($this->get('security.authorization_checker')->isGranted('ROLE_SUPER_ADMIN')) {
            $courseType = new CourseType();
        } else {
            $courseType = new CourseType($this->getUser()->getCompany());
        }
        $form = $this->createForm($courseType, $entity, array(
            'action' => $this->generateUrl('backend_course_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing Course entity.
     *
     * @Route("/{id}", name="backend_course_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\Course:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("edit_course");
        $em = $this->getDoctrine()->getManager();

        $categoryPresencial = $this->getCategoryRepository()->findByModality('Presencial');
        $categoryEad = $this->getCategoryRepository()->findByModality('EAD');

        $entity = $em->getRepository('AppBundle:Course')->find($id);
        $userGroups = $this->getUserGroupRepository()->findAll();

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            try {
                if($entity->getPontuation() && !$entity->getSalutarisId()){
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Para cursos com <b>Pontuação</b> você deve preencher o <b>Id no Salutaris</b>!');
                    return array(
                        'entity'      => $entity,
                        'userGroups' => $userGroups,
                        'edit_form'   => $editForm->createView(),
                    );
                }
                if($entity->getSalutarisId() && !$entity->getPontuation()){
                    $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Para cursos com <b>Id no Salutaris</b> você deve preencher a <b>Pontuação</b>!');
                    return array(
                        'entity'      => $entity,
                        'userGroups' => $userGroups,
                        'edit_form'   => $editForm->createView(),
                    );
                }
                $em->flush();

                //TODO: apagar registros antigos
                foreach ($entity->getCourseAccessControls() as $courseAccessControls) {
                    $em->remove($courseAccessControls);
                }

                $autoInscriptionOptions = $request->request->get('autoInscription');
                if(!empty($autoInscriptionOptions)){
                    foreach ($autoInscriptionOptions as $key => $value) {
                        $courseAccessControl = new CourseAccessControl();
                        $courseAccessControl->setCourse($entity);
                        $courseAccessControl->setAccess($value);
                        $em->persist($courseAccessControl);
                        $courseAccessControl->setAccess($value);
                    }
                }
                $em->flush();

                //Salva novos registros
                $request->getSession()
                 ->getFlashBag()
                 ->add('success', 'Registro atualizado com sucesso!');

                return $this->redirect($this->generateUrl('backend_course', array('id' => $id)));
            } catch (\Exception $e) {
                $this->log($e);
                $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity'      => $entity,
            'userGroups' => $userGroups,
            'edit_form'   => $editForm->createView(),
            'categoryPresencial'=> $categoryPresencial,
            'categoryEad'=> $categoryEad,
        );
    }

    /**
     * Deletes a Course entity.
     *
     * @Route("/{id}/delete", name="backend_course_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("remove_course");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($id);

        if (!$course) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $canRemove = true;
        $msgError = 'Não é possível remover esse curso devido o(s) seguinte(s) erro(s): <br/>';

        if ($course->getTotalLessons()) {
            $msgError .= '- Possui '.$course->getTotalLessons().' aula(s) associada(s). <br/>';
            $canRemove = false;
        }
        if ($course->getTotalEvaluations()) {
            $msgError .= '- Possui '.$course->getTotalEvaluations().' avaliações. <br/>';
            $canRemove = false;
        }
        if (count($course->getFeedbacks())) {
            $msgError .= '- Possui '.count($course->getFeedbacks()).' feedback dos usuarios. <br/>';
            $canRemove = false;
        }
        if (count($course->getSurveyanswers())) {
            $msgError .= '- Possui '.count($course->getSurveyanswers()).' pesquisa(s) de satisfação. <br/>';
            $canRemove = false;
        }


        try {
            if ($canRemove) {
                $em->remove($course);
                $course->unlinkImages();
                $em->flush();
                $this->successMessage($request, 'Registro excluído com sucesso!');
            } else {
                $this->warningMessage($request, $msgError);
            }
        } catch (\Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course'));
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription", name="backend_course_inscriptions")
     * @Method("GET")
     * @Template()
     */
    public function inscriptionsAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_all");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'entity'  => $entity,
            'id' => $id
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/pesquisas-de-satisfacao", name="backend_course_survey")
     * @Method("GET")
     * @Template()
     */
    public function courseSurveyAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_survey_course");
        $em = $this->getDoctrine()->getManager();
        
        $course = $em->getRepository('AppBundle:Course')->find($id);

        if (!$course) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $surveyAnswers = $em->getRepository('AppBundle:SurveyAnswer')->findAnswersGroupedBy($course);
        return array(
            'course' => $course,
            'surveyAnswers' => $surveyAnswers
        );
    }

    /**
     * @Route("/inscription/{course_id}/menu-options", name="course_menu_options")
     */
    public function inscriptionMenuOptionAction($course_id)
    {
        $course = $this->getCourseRepository()->find($course_id);
        $userGroups = $this->getUserGroupRepository()->findAll();
        return $this->render('AppBundle:Backend\Course:menu_options_index.html.twig', array(
            'course' => $course,
            'userGroups' => $userGroups
        ));
    }

    /**
     * Lists all Course entities.
     *
     * @Route("/export/{id}/pesquisas-de-satisfacao-{slug}", name="backend_course_survey_export")
     * @Method("GET")
     * @Template()
     */
    public function surveyExportAction(Request $request, $id, $slug)
    {
        //excel
        $phpExcelObject = $this->get('phpexcel')->createPHPExcelObject();
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($id);

        if (!$course) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $phpExcelObject->getProperties()->setCreator($this->getUser()->getUsername())
            ->setLastModifiedBy("Unimed")
            ->setTitle("Pesquisa de Satisfação (Unimed)")
            ->setSubject("Pesquisa de Satisfação (Unimed)")
            ->setDescription("Pesquisa de Satisfação (Unimed)")
            ->setKeywords("Unimed")
            ->setCategory("Pesquisa de Satisfação ");

        //Primeira aba da planilha - Certidões vencidas
        $phpExcelObject->setActiveSheetIndex(0)
        ->setCellValue('A1', 'Pesquisa de Satisfação')
        ->setCellValue('B1', 'Respondente')
        ->setCellValue('C1', 'Pergunta')
        ->setCellValue('D1', 'Resposta')
        ->setCellValue('E1', 'Data')
        ->setCellValue('F1', 'Comentário');
        $i = 2;
        foreach ($course->getSurveyanswers() as $surveyanswers) {
            $phpExcelObject->setActiveSheetIndex(0)
                ->setCellValue('A'.$i, $surveyanswers->getSurvey())
                ->setCellValue('B'.$i, $surveyanswers->getUser())
                ->setCellValue('C'.$i, $surveyanswers->getSurveyItem())
                ->setCellValue('D'.$i, $surveyanswers->getValue())
                ->setCellValue('E'.$i, $surveyanswers->getDtCreation()->format('d/m/Y'))
                ->setCellValue('F'.$i, $surveyanswers->getComment());
            $i++;
        }

        //Autosize nas colunas da planilha
        foreach (range('A', 'F') as $columnID) {
            $phpExcelObject->getActiveSheet()->getColumnDimension($columnID)
            ->setAutoSize(true);
        }

        $phpExcelObject->setActiveSheetIndex(0);

        $phpExcelObject->getActiveSheet()->setTitle('Inscrições');
    
        $writer = $this->get('phpexcel')->createWriter($phpExcelObject, 'Excel2007');
    
        $response = $this->get('phpexcel')->createStreamedResponse($writer);
    
        $dispositionHeader = $response->headers->makeDisposition(
            ResponseHeaderBag::DISPOSITION_ATTACHMENT,
            'Unimed - Acompanhamento.xlsx'
        );
        $response->headers->set('Content-Type', 'text/vnd.ms-excel; charset=utf-8');
        $response->headers->set('Pragma', 'public');
        $response->headers->set('Cache-Control', 'maxage=1');
        $response->headers->set('Content-Disposition', $dispositionHeader);
    
        return $response;
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/feedbacks", name="backend_course_feedbacks")
     * @Method("GET")
     * @Template()
     */
    public function courseFeedbackAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_feedback_course");
        $em = $this->getDoctrine()->getManager();
        
        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $entities = $em->getRepository('AppBundle:CourseFeedback')->findBy(array("course"=>$entity, 'checked'=>'0'));

        return array(
            'entities' => $entities,
            'entity' => $entity,
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/feedbacks/store/{feedback_id}", name="backend_course_feedbacks_store")
     * @Method("GET")
     * @Template()
     */
    public function courseFeedbackStoreAction(Request $request, $id, $feedback_id)
    {
        $this->verificationUserAccessControl("index_feedback_course");
        $em = $this->getDoctrine()->getManager();
        
        $entity = $em->getRepository('AppBundle:CourseFeedback')->find($feedback_id);
        $course = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $entity->setChecked(true);
        $em->persist($entity);
        $em->flush();

        return $this->redirect($this->generateUrl('backend_course_feedbacks', array('id' => $course->getId())));
    }


    public function getCountInscriptionFromCourseId($courseId)
    {
        $repositoryInscription = $this->getInscriptionRepository();
        $inscriptionStatus = new InscriptionStatus();
        $countInscriptions = $repositoryInscription->findTotalActiveInscriptions($courseId, $inscriptionStatus::APROVADO);

        return $countInscriptions;
    }

    public function getCountClients()
    {
        $clientRepository = $this->getClientRepository();
        $countClients = $clientRepository->findTotalClients();

        return $countClients;
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/setor", name="backend_course_inscriptions_sector")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsSector.html.twig")
     */
    public function inscriptionsSectorAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $sector = $em->getRepository('AppBundle:Sector')->findAll();

        //paginator sector
        $paginatorsector  = $this->get('knp_paginator');
        $paginationsector = $paginatorsector->paginate($sector, $request->query->get('page', 1), 10);


        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'sector' => $paginationsector,
            'entity'  => $entity,
            'query' => null
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/setor/search", name="backend_course_inscriptions_sector_search")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsSector.html.twig")
     */
    public function inscriptionsSectorSearchAction(Request $request, $id, $query = null)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();
        $query = $this->getRequest()->get("query");

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $sector = $em->getRepository('AppBundle:Sector')->findForSearch($query);

        //paginator sector
        $paginatorsector  = $this->get('knp_paginator');
        $paginationsector = $paginatorsector->paginate($sector, $request->query->get('page', 1), 10);

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'sector' => $paginationsector,
            'entity'  => $entity,
            'query' => $query,
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/cargos", name="backend_course_inscriptions_cargos")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsRegistration.html.twig")
     */
    public function inscriptionsCargoAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $jobrole = $em->getRepository('AppBundle:JobRole')->findAll();

        //paginator jobrole
        $paginatorjobrole  = $this->get('knp_paginator');
        $paginationjobrole = $paginatorjobrole->paginate($jobrole, $request->query->get('page', 1), 10);

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'jobrole' => $paginationjobrole,
            'entity'  => $entity,
            'query' => null
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/all", name="backend_course_inscriptions_all")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsAll.html.twig")
     */
    public function inscriptionsAllAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_active");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'entity'  => $entity,
        );
    }

    /**
     * Realize import for inscriptions.
     *
     * @Route("/{id}/inscription/import", name="backend_course_inscriptions_import")
     * @Template("AppBundle:Backend\Course:inscriptionsImport.html.twig")
     */
    public function inscriptionsImportAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_active");
        $em = $this->getDoctrine()->getManager();
        
        $entity = $em->getRepository('AppBundle:Course')->find($id);
        
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }
        $inscriptionImport = new InscriptionImport();
        $form = $this->createForm(new InscriptionImportType(), $inscriptionImport);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->log('[inscriptionsImportAction] PROCESSANDO O ARQUIVO');

            $em = $this->getDoctrine()->getManager();

            $inputFileName = $inscriptionImport->getFile();

            try {
                $inputFileType = \PHPExcel_IOFactory::identify($inputFileName);
                $objReader = \PHPExcel_IOFactory::createReader($inputFileType);
                $objPHPExcel = $objReader->load($inputFileName);
            } catch (\Exception $e) {
                // die('Error loading file "' . pathinfo($inputFileName, PATHINFO_BASENAME) . '": ' . $e->getMessage());
                $this->log($e);
                $this->errorMessage($request, 'Ocorreu um erro ao processar o arquivo. Verifique o padrão utilizado e tente novamente.');
            }

            //  Get worksheet dimensions
            $sheet = $objPHPExcel->getSheet(0);
            $highestRow = $sheet->getHighestRow();
            $highestColumn = $sheet->getHighestColumn();

            $total = 0;

            try {
                $this->log('[inscriptionsImportAction] ##################');
                $this->log('[inscriptionsImportAction] Loop nas linhas do arquivo');
                $inscriptionService = $this->container->get('inscription.service');
                //  Loop through each row of the worksheet in turn
                for ($row = 2; $row <= $highestRow; $row++) {
                    //  Read a row of data into an array
                    $rowData = $sheet->rangeToArray('A' . $row . ':' . $highestColumn . $row, null, true, false);
                    //Obtem o registro da linha
                    $userRow = $rowData[0];

                    if(($userRow[0] && $userRow[1] && $userRow[2]) || $userRow[3]){
                        $userItem = $em->getRepository('AppBundle:Client')->findOneBy(
                            array(
                                "mat_colaborador" => $userRow[0],
                                "num_enterprise" => $userRow[1],
                                "tip_colaborador" => $userRow[2]
                            )
                        );
                        if(!$userItem){
                            $userItem = $em->getRepository('AppBundle:Client')->findOneBy(
                                array(
                                    "login" => $userRow[3]
                                )
                            );
                        }

                        if(!$userItem){
                            $this->errorMessage($request, "<b>Registro da linha ".$row." não encontrado.</b> <br>");
                        }else{
                            if($userItem->isEnabled()){
                                $userInscription = $em->getRepository('AppBundle:Inscription')->getLastInscriptionByUser($userItem, $entity);
                                $inscription = null;
                                if (!$userInscription) {
                                    $attempts = null;
                                    $inscription = $inscriptionService->createInscriptionByInscriptionStatusApproved($userItem, $entity, $attempts);
                                } elseif ($userInscription->getInscriptionStatus()->getId() == InscriptionStatus::CANCELADO) {
                                    $attempts = null;
                                    $inscription = $inscriptionService->createInscriptionByInscriptionStatusApproved($userItem, $entity, $attempts);
                                }else{
                                    $this->errorMessage($request, "<b>".$userItem->getFullName()."</b> da <b>linha ".$row."</b> já está matriculado no curso <b>".$entity->getName()."</b>.<br>");
                                }
        
                                if ($inscription) {
                                    $em->persist($inscription);
                                }
                            }else{
                                $this->errorMessage($request, "<b>".$userItem->getFullName()."</b> da <b>linha ".$row."</b> está inativo.<br>");
                            }
                        }
                    }
                    $em->flush();
                    $this->log($userRow);
                }
            } catch (\Exception $e) {
                $this->log("ERRO: ");
                $this->log($e);
                $this->get('session')->getFlashBag()->add('error', "Ocorreu um erro ao realizar a importacão.");
                // die('Error loading file "' . pathinfo($inputFileName, PATHINFO_BASENAME) . '": ' . $e->getMessage());
            }

            $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Importação de matrículas realizada com sucesso.');
        }

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'entity'  => $entity,
            'form' => $form->createView(),
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/group-user/{group_user}", name="backend_course_inscriptions_group_user")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsGroupUser.html.twig")
     */
    public function inscriptionByGroupUserAction(Request $request, $id, $group_user)
    {
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($id);
        $clientService = $this->container->get('client.service');

        if (!$course) {
            throw $this->createNotFoundException('Curso não encontrado.');
        }
        
        $userGroup = $this->getUserGroupRepository()->find($group_user);

        $clients = $this->getClientRepository()->findByUsergroup($userGroup);

        // $clients = $clientService->getAllClientByGroupUser($group_user);

        $paginator = $this->get('knp_paginator');
        $paginator = $paginator->paginate($clients, $request->query->get('page', 1), 10);

        return array(
            'course' => $course,
            'clients' => $paginator,
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'group_user' => $group_user,
            'query' => null
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/group-user/{group_user}/register", name="backend_course_inscriptions_group_user_register")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsGroupUser.html.twig")
     */
    public function inscriptionInMassByGroupUserAction(Request $request, $id, $group_user)
    {
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($id);

        $inscriptionService = $this->container->get('inscription.service');
        // $clientService = $this->container->get('client.service');
        // $clients = $clientService->getAllClientByGroupUser($group_user);
        $userGroup = $this->getUserGroupRepository()->find($group_user);
        $clients = $this->getClientRepository()->findByUsergroup($userGroup);

        $inscriptionService->incriptionInMassUsers($clients, $course);

        $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Matrículas realizadas com sucesso.');

        return $this->redirectToRoute('backend_course_inscriptions_group_user', array(
            'id' => $id,
            'group_user' => $group_user,
            'query' => null
        ));
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/cargos/search", name="backend_course_inscriptions_cargos_search")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsRegistration.html.twig")
     */
    public function inscriptionsCargoSearchAction(Request $request, $id, $query = null)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();

        $query = $this->getRequest()->get("query");

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $jobrole = $em->getRepository('AppBundle:JobRole')->findForSearch($query);

        //paginator jobrole
        $paginatorjobrole  = $this->get('knp_paginator');
        $paginationjobrole = $paginatorjobrole->paginate($jobrole, $request->query->get('page', 1), 10);

        return array(
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'jobrole' => $paginationjobrole,
            'entity'  => $entity,
            'query' => $query,
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/matricula", name="backend_course_inscriptions_matricula")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsClients.html.twig")
     */
    public function inscriptionsRegistrationction(Request $request, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_client");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $clients = $em->getRepository('AppBundle:Client')->findByPagination();

        //paginator client
        $paginatorclient  = $this->get('knp_paginator');
        $paginationclient = $paginatorclient->paginate($clients, $request->query->get('page', 1), 10);


        return array(
            'clients' => $paginationclient,
            'entity'  => $entity,
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'query' => null
        );
    }

    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{id}/inscription/matricula/search", name="backend_course_inscriptions_matricula_search")
     * @Method("GET")
     * @Template("AppBundle:Backend\Course:inscriptionsClients.html.twig")
     */
    public function inscriptionsRegistrationSearchction(Request $request, $id, $query = null)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_client");
        $em = $this->getDoctrine()->getManager();
        $query = $this->getRequest()->get("query");

        $entity = $em->getRepository('AppBundle:Course')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Course entity.');
        }

        $clients = $em->getRepository('AppBundle:Client')->findByPagination($query);

        //paginator client
        $paginatorclient  = $this->get('knp_paginator');
        $paginationclient = $paginatorclient->paginate($clients, $request->query->get('page', 1), 10);


        return array(
            'clients' => $paginationclient,
            'entity'  => $entity,
            'count_inscriptions' => $this->getCountInscriptionFromCourseId($id),
            'count_clients' => $this->getCountClients(),
            'query' => $query,
        );
    }

    /**
     * Activate Inscription the a User.
     *
     * @Route("/course/{course_id}/activate/inscription/{user_id}/{group_user}", name="backend_inscription_activate")
     *
     */
    public function activateInscriptionAction(Request $request, $course_id, $user_id, $group_user = '')
    {
        $this->verifyAdmin();
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $user = $em->getRepository('UserBundle:User')->find($user_id);
        $inscriptionService = $this->container->get('inscription.service');

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$user) {
            throw $this->createNotFoundException('Registro não encontrado de Funcionário.');
        }

        try {
            $userInscription = $em->getRepository('AppBundle:Inscription')->getLastInscriptionByUser($user, $course);
            $inscription = null;
            if (!$userInscription) {
                $attempts = null;
                $inscription = $inscriptionService->createInscriptionByInscriptionStatusApproved($user, $course, $attempts);
            } elseif ($userInscription->getInscriptionStatus()->getId() == InscriptionStatus::CANCELADO) {
                $attempts = null;
                $inscription = $inscriptionService->createInscriptionByInscriptionStatusApproved($user, $course, $attempts);
            }

            if ($inscription) {
                $em->persist($inscription);
                $em->flush();

                $this->successMessage($request, 'Registro liberado com sucesso!');
            }
        } catch (\Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        if ($group_user) {
            return $this->redirect($this->generateUrl('backend_course_inscriptions_group_user', array(
                "id" => $course->getId(),
                "group_user" => $group_user,
            )));
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions_matricula', array(
            "id" => $course->getId()
        )));
    }


    /**
     * Lists all inscriptions entities.
     *
     * @Route("/{inscription_id}/change/{attempts}/{screen}", name="backend_change_attempts")
     */
    public function changeAttemptsAction(Request $request, $inscription_id, $attempts, $screen = "funcionario")
    {
        $em = $this->getDoctrine()->getManager();
        $inscription = $em->getRepository('AppBundle:Inscription')->find($inscription_id);
        $inscription->setAttempts($attempts);
        $em->flush();
        if ($screen == "all") {
            return $this->redirect($this->generateUrl('backend_course_inscriptions_all', array("id" => $inscription->getCourse()->getId())));
        } else {
            return $this->redirect($this->generateUrl('backend_course_inscriptions_matricula', array("id" => $inscription->getCourse()->getId())));
        }
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/course/{course_id}/activate-sector/inscription/{sector_id}/sector", name="backend_inscription_activate_sector")
     *
     */
    public function activateInscriptionSectorAction(Request $request, $course_id, $sector_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $sector = $em->getRepository('AppBundle:Sector')->find($sector_id);
        $user = $em->getRepository('UserBundle:User')->findBySector($sector);

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$user) {
            $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Nenhum funcionário cadastrado nesse setor');
            return $this->redirect($this->generateUrl('backend_course_inscriptions', array("id" => $course->getId())));
        }

        try {
            $verify = 0;
            foreach ($user as $us) {
                $userInscription = $em->getRepository('AppBundle:Inscription')->getLastInscriptionByUser($us, $course);
                if ($userInscription == null) {
                    $inscription = new Inscription();
                    $inscription->setUser($us);
                    $inscription->setCourse($course);
                    if ($course->getDisponibleDays()!=null) {
                        ///Instancia um novo objeto de matrícula
                        $dtEnd = new \DateTime();
                        if ($course->getDisponibleDays() > 730) {
                            $dtEnd->add(new \DateInterval('P730D'));
                        } else {
                            $dtEnd->add(new \DateInterval('P'. $course->getDisponibleDays() .'D'));
                        }
                        $dtBegin = new \DateTime();
                        $inscription->setDtBegin($dtBegin);
                        $inscription->setDtEnd($dtEnd);
                    }
                    $inscription->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
                    $em->persist($inscription);
                    $em->flush();
                    $verify = 1;
                } elseif ($userInscription->getInscriptionStatus()->getId() == InscriptionStatus::CANCELADO) {
                    $inscription = new Inscription();
                    $inscription->setUser($us);
                    $inscription->setCourse($course);
                    if ($course->getDisponibleDays()!=null) {
                        ///Instancia um novo objeto de matrícula
                        $dtEnd = new \DateTime();
                        if ($course->getDisponibleDays() > 730) {
                            $dtEnd->add(new \DateInterval('P730D'));
                        } else {
                            $dtEnd->add(new \DateInterval('P'. $course->getDisponibleDays() .'D'));
                        }
                        $dtBegin = new \DateTime();
                        $inscription->setDtBegin($dtBegin);
                        $inscription->setDtEnd($dtEnd);
                    }
                    $inscription->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
                    $em->remove($userInscription);
                    $em->persist($inscription);
                    $em->flush();
                    $verify = 1;
                }
            }
            if ($verify == 1) {
                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro liberado com sucesso!');
            } else {
                $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Nenhum funcionário desse setor para ser cadastrado');
            }
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions_sector', array("id" => $course->getId())));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/course/{course_id}/activate-jobrole/inscription/{jobrole_id}/jobrole", name="backend_inscription_activate_jobrole")
     *
     */
    public function activateInscriptionJobRoleAction(Request $request, $course_id, $jobrole_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $job = $em->getRepository('AppBundle:JobRole')->find($jobrole_id);
        $user = $em->getRepository('UserBundle:User')->findByJobRole($job);

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$user) {
            $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Nenhum funcionário cadastrado nesse setor');
            return $this->redirect($this->generateUrl('backend_course_inscriptions', array("id" => $course->getId())));
        }

        try {
            $verify = 0;
            foreach ($user as $us) {
                $userInscription = $em->getRepository('AppBundle:Inscription')->getLastInscriptionByUser($us, $course);
                if ($userInscription == null) {
                    $inscription = new Inscription();
                    $inscription->setUser($us);
                    $inscription->setCourse($course);
                    if ($course->getDisponibleDays()!=null) {
                        ///Instancia um novo objeto de matrícula
                        $dtEnd = new \DateTime();
                        if ($course->getDisponibleDays() > 730) {
                            $dtEnd->add(new \DateInterval('P730D'));
                        } else {
                            $dtEnd->add(new \DateInterval('P'. $course->getDisponibleDays() .'D'));
                        }
                        $dtBegin = new \DateTime();
                        $inscription->setDtBegin($dtBegin);
                        $inscription->setDtEnd($dtEnd);
                    }
                    $inscription->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
                    $em->persist($inscription);
                    $em->flush();
                    $verify = 1;
                } elseif ($userInscription->getInscriptionStatus()->getId() == InscriptionStatus::CANCELADO) {
                    $inscription = new Inscription();
                    $inscription->setUser($us);
                    $inscription->setCourse($course);
                    if ($course->getDisponibleDays()!=null) {
                        ///Instancia um novo objeto de matrícula
                        $dtEnd = new \DateTime();
                        if ($course->getDisponibleDays() > 730) {
                            $dtEnd->add(new \DateInterval('P730D'));
                        } else {
                            $dtEnd->add(new \DateInterval('P'. $course->getDisponibleDays() .'D'));
                        }
                        $dtBegin = new \DateTime();
                        $inscription->setDtBegin($dtBegin);
                        $inscription->setDtEnd($dtEnd);
                    }
                    $inscription->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
                    $em->remove($userInscription);
                    $em->persist($inscription);
                    $em->flush();
                    $verify = 1;
                }
            }
            if ($verify == 1) {
                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro liberado com sucesso!');
            } else {
                $request->getSession()
                   ->getFlashBag()
                   ->add('error', 'Nenhum funcionário desse setor para ser cadastrado');
            }
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions_cargos', array("id" => $course->getId())));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/course/{course_id}/cancel/inscription/{id}/{group_user}", name="backend_inscription_cancel")
     *
     */
    public function cancelInscriptionAction(Request $request, $course_id, $id, $group_user = '')
    {
        $this->verificationUserAccessControl("index_inscription_course_all");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $entity = $em->getRepository('AppBundle:Inscription')->find($id);

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$entity) {
            throw $this->createNotFoundException('Registro não encontrado de Matrícula.');
        }

        try {
            $entity->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::CANCELADO));
            $em->persist($entity);
            $em->flush();
            $request->getSession()
               ->getFlashBag()
               ->add('success', 'Registro cancelado com sucesso!');
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        if ($group_user) {
            return $this->redirect($this->generateUrl('backend_course_inscriptions_group_user', array(
                "id" => $course->getId(),
                "group_user" => $group_user,
            )));
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions_matricula', array("id" => $course->getId())));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/course/{course_id}/cancel/inscription/{sector_id}/sector", name="backend_inscription_cancel_sector")
     *
     */
    public function cancelInscriptionSectorAction(Request $request, $course_id, $sector_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_sector");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $sector = $em->getRepository('AppBundle:Sector')->find($sector_id);
        $user = $em->getRepository('UserBundle:User')->findBySector($sector);
        foreach ($user as $us) {
            $entity = $em->getRepository('AppBundle:Inscription')->findByUser($user);
        }

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$entity) {
            throw $this->createNotFoundException('Registro não encontrado de Matrícula.');
        }

        try {
            foreach ($entity as $entities) {
                $entities->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::CANCELADO));
                $em->persist($entities);
                $em->flush();
                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro cancelado com sucesso!');
            }
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions_sector', array("id" => $course->getId())));
    }

    /**
     * Cancels a Inscription entity.
     *
     * @Route("/course/{course_id}/cancel/inscription/{jobrole_id}/jobrole", name="backend_inscription_cancel_jobrole")
     *
     */
    public function cancelInscriptionJobRoleAction(Request $request, $course_id, $jobrole_id)
    {
        $this->verificationUserAccessControl("index_inscription_course_by_jobrole");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $job = $em->getRepository('AppBundle:JobRole')->find($jobrole_id);
        $user = $em->getRepository('UserBundle:User')->findByJobRole($job);
        foreach ($user as $us) {
            $entity = $em->getRepository('AppBundle:Inscription')->findByUser($user);
        }

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$entity) {
            throw $this->createNotFoundException('Registro não encontrado de Matrícula.');
        }

        try {
            foreach ($entity as $entities) {
                $entities->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::CANCELADO));
                $em->persist($entities);
                $em->flush();
                $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro cancelado com sucesso!');
            }
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions_cargos', array("id" => $course->getId())));
    }

    /**
     * Deletes a Inscription entity.
     *
     * @Route("/course/{course_id}/delete/inscription/{id}", name="backend_inscription_delete")
     *
     */
    public function deleteInscriptionAction(Request $request, $course_id, $id)
    {
        $this->verificationUserAccessControl("index_inscription_course_all");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course_id);
        $entity = $em->getRepository('AppBundle:Inscription')->find($id);

        if (!$course) {
            throw $this->createNotFoundException('Registro não encontrado de Curso.');
        }

        if (!$entity) {
            throw $this->createNotFoundException('Registro não encontrado de Matrícula.');
        }

        try {
            $em->remove($entity);
            $em->flush();
            $request->getSession()
               ->getFlashBag()
               ->add('success', 'Registro excluído com sucesso!');
        } catch (\Exception $e) {
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_course_inscriptions', array("id" => $course->getId())));
    }

    /**
     *
     * @Route("/course/{course_id}/survey-answers/{user_id}", name="course_user_survey_answers", options={"expose"=true})
     */
    public function courseUserSurveyAnswers(Request $request, $course_id, $user_id)
    {
        $em   = $this->getDoctrine()->getManager();
        $user = $em->getRepository('UserBundle:User')->findOneById($user_id);
        $course = $em->getRepository('AppBundle:Course')->findOneById($course_id);

        $surveyAnswers = $em->getRepository('AppBundle:SurveyAnswer')->findBy(array('course'=>$course, 'user'=>$user));
        $return = array("responseCode" => 200, "results" => $surveyAnswers, 'survey' => $course->getSurvey()->getName(), 'user'=> $user->getFullName());
        return $this->returnJson($return);
    }

    /**
     *
     * @Route("/course/{course_id}/cancel/{user_id}", name="course_cancel_subscribe", options={"expose"=true})
     */
    public function cancelInscription(Request $request, $course_id, $user_id)
    {
        $em   = $this->getDoctrine()->getManager();
        $user = $em->getRepository('UserBundle:User')->findOneById($user_id);
        $course = $em->getRepository('AppBundle:Course')->findOneById($course_id);

        if (!$user) {
            $this->warningMessage($request, 'Usuario não encontrado');
            return new Response(json_encode(array(
                'status' => 500,
                'response' => 'done',
                'msg' => 'Usuario não encontrado'
            )));
        }

        $inscriptionStatusAprovado = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO);
        $inscriptionStatusCancelado = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::CANCELADO);

        $inscription = $em->getRepository('AppBundle:Inscription')->findOneBy(array(
            'user' => $user,
            'course' => $course,
            'inscription_status' => $inscriptionStatusAprovado
        ));

        if (!$inscription) {
            $msg = 'Curso não encontrado';
            $status = 500;
        } else {
            $status = 200;
            $msg = 'Rematrícula realizada com sucesso';
            $inscription->setInscriptionStatus($inscriptionStatusCancelado);
            $em->persist($inscription);
            
            $newInscription = new Inscription();
            $newInscription->setUser($inscription->getuser());
            $newInscription->setCourse($inscription->getCourse());
            if ($inscription->getCourse()->getDisponibleDays()!=null) {
                ///Instancia um novo objeto de matrícula
                $dtEnd = new \DateTime();
                if ($inscription->getCourse()->getDisponibleDays() > 730) {
                    $dtEnd->add(new \DateInterval('P730D'));
                } else {
                    $dtEnd->add(new \DateInterval('P'. $inscription->getCourse()->getDisponibleDays() .'D'));
                }
                $dtBegin = new \DateTime();
                $newInscription->setDtBegin($dtBegin);
                $newInscription->setDtEnd($dtEnd);
            }
            $newInscription->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
            $em->persist($newInscription);
            $em->flush();
        }

        return new Response(json_encode(array(
            'status' => $status,
            'response' => 'done',
            'msg' => $msg
        )));
    }
}
