<?php

namespace AppBundle\Controller\Backend;

use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Evaluation;
use AppBundle\Form\EvaluationType;

/**
 * Evaluation controller.
 *
 * @Route("/backend/evaluation")
 */
class EvaluationController extends BaseController
{

  /**
   * Displays a form to create a new Evaluation entity.
   *
   * @Route("/new/{module}", name="backend_evaluation_new")
   * @Method("GET")
   * @Template()
   */
    public function newAction($module)
    {
        $this->verificationUserAccessControl("new_evaluation");
        $em = $this->getDoctrine()->getManager();
        $module = $em->getRepository('AppBundle:Module')->find($module);

        if (!$module) {
            throw $this->createNotFoundException('Módulo não encontrado.');
        }

        $entity = new Evaluation();
        $form   = $this->createCreateForm($entity);

        return array(
            'module' => $module,
            'entity' => $entity,
            'questions' => null,
            'form'   => $form->createView(),
        );
    }

    /**
     * Lists all Evaluation entities.
     *
     * @Route("/{module}", name="backend_evaluation")
     * @Method("GET")
     * @Template()
     */
    public function indexAction(Request $request, $module)
    {
        $this->verificationUserAccessControl("index_evaluation");
        $em = $this->getDoctrine()->getManager();

        $module = $em->getRepository('AppBundle:Module')->find($module);

        if (!$module) {
            throw $this->createNotFoundException('Módulo não encontrado.');
        }

        $entities = $em->getRepository("AppBundle:Evaluation")->findBy(['module' => $module]);

        return array(
            'module' => $module,
            'entities' => $entities,
        );
    }
    /**
     * Creates a new Evaluation entity.
     *
     * @Route("/{module}", name="backend_evaluation_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\Evaluation:new.html.twig")
     */
    public function createAction(Request $request, $module = null)
    {
        $this->verificationUserAccessControl("new_evaluation");
        $em = $this->getDoctrine()->getManager();
        $entity = new Evaluation();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);
        $module = $em->getRepository('AppBundle:Module')->find($module);
        if (!$module) {
            throw $this->createNotFoundException('Módulo não encontrado.');
        }
        if ($form->isValid()) {
            try {
                $entity->setModule($module);
                $em->persist($entity);
                $em->flush();
                $request->getSession()->getFlashBag()->add('success', 'Registro criado com sucesso!');
                return $this->redirect($this->generateUrl('backend_evaluation', array('module' => $module->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()->getFlashBag()->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }
        return array(
            'module' => $module,
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Creates a form to create a Evaluation entity.
     * @param Evaluation $entity The entity
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Evaluation $entity)
    {
        $this->verificationUserAccessControl("new_evaluation");
        $form = $this->createForm(new EvaluationType(), $entity, array(
            'action' => $this->generateUrl('backend_evaluation_create'),
            'method' => 'POST',
        ));
        $form->add('submit', 'submit', array('label' => 'Criar'));
        return $form;
    }

    /**
     * Displays a form to edit an existing Evaluation entity.
     * @Route("/{id}/edit", name="backend_evaluation_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl("edit_evaluation");
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Evaluation')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Evaluation entity.');
        }
        $module = $entity->getModule();
        //Envio o questions para o twig para verificar a quantidade de questões que a avaliação possui
        $questions = $em->getRepository("AppBundle:Question")->findBy(['evaluation' => $entity]);
        $editForm = $this->createEditForm($entity);
        return array(
            'module'      => $module,
            'entity'      => $entity,
            'questions'   => $questions,
            'edit_form'   => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a Evaluation entity.
    * @param Evaluation $entity The entity
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Evaluation $entity)
    {
        $this->verificationUserAccessControl("edit_evaluation");
        $form = $this->createForm(new EvaluationType(), $entity, array(
            'action' => $this->generateUrl('backend_evaluation_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));
        $form->add('submit', 'submit', array('label' => 'Atualizar'));
        return $form;
    }
    /**
     * Edits an existing Evaluation entity.
     * @Route("/{id}", name="backend_evaluation_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\Evaluation:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("edit_evaluation");
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Evaluation')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Evaluation entity.');
        }
        $module = $entity->getModule();
        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);
        if ($editForm->isValid()) {
            try {
                $em->flush();
                $request->getSession()->getFlashBag()->add('success', 'Registro atualizado com sucesso!');
                return $this->redirect($this->generateUrl('backend_evaluation', array('module' => $module->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()->getFlashBag()->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }
        return array(
            'module'      => $module,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }
    /**
     * Deletes a Evaluation entity.
     *
     * @Route("/{id}/delete", name="backend_evaluation_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("remove_evaluation");
        $em = $this->getDoctrine()->getManager();
        $evaluation = $em->getRepository('AppBundle:Evaluation')->find($id);
        if (!$evaluation) {
            throw $this->createNotFoundException('Unable to find Evaluation entity.');
        }
        $module = $evaluation->getModule();

        $canRemove = true;
        $msgError = 'Não é possível remover essa avaliação devido o(s) seguinte(s) erro(s): <br/>';

        $userEvaluations = $em->getRepository('AppBundle:UserEvaluation')->findByEvaluation($evaluation);
        if (count($userEvaluations)) {
            $msgError .= '- Possui ' . count($userEvaluations) . ' avaliação(ões) associada(s) à avaliação selecionada. <br/>';
            $canRemove = false;
        }

        if (count($evaluation->getQuestions())) {
            $msgError .= '- Possui ' . count($evaluation->getQuestions()) . ' questão(ões) associada(s) à avaliação selecionada. <br/>';
            $canRemove = false;
        }

        if (count($evaluation->getQuestions())) {
            $msgError .= '- Possui ' . count($evaluation->getQuestions()) . ' questão(ões) associada(s) à avaliação selecionada. <br/>';
            $canRemove = false;
        }
        
        try {
            if ($canRemove) {
                $em->remove($evaluation);
                $em->flush();
                $this->successMessage($request, 'Registro excluído com sucesso!');
            } else {
                $this->warningMessage($request, $msgError);
            }
        } catch (\Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_evaluation', array('module' => $module->getId())));
    }
}
