<?php

namespace AppBundle\Controller\Backend;

use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Lesson;
use AppBundle\Entity\Scorm12Sco;
use AppBundle\Form\LessonType;
use Symfony\Component\HttpKernel\Exception;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Lesson controller.
 *
 * @Route("/backend/lesson")
 */
class LessonController extends BaseController
{

  /**
   * @Route("/lesson_sort_ajax_list", name="lesson_sort_ajax_list", options={"expose"=true})
   * @Template()
   */
    public function sortAjaxListAction()
    {
        try {
            $em = $this->getDoctrine()->getManager();
            $request = Request::createFromGlobals();
            $lessonsOrder = $request->request->get('lesson_order');

            $listingCounter = 1;
            $lessonsOrder = explode("&", $lessonsOrder);
            foreach ($lessonsOrder as $lessonId) {
                $lessonId = str_replace("lesson_order[]=", "", $lessonId);
                $lesson = $em->getRepository('AppBundle:Lesson')->find($lessonId);
                $lesson->setPosition($listingCounter);
                $em->persist($lesson);
                $em->flush();
                $listingCounter++;
            }

            $return = array("responseCode" => 200, "results" => $lessonsOrder);
        } catch (\Exception $e) {
            $return = array("responseCode" => 400, 'errorMessage' => $e->getTraceAsString());
        }

        return $this->returnJson($return);
    }

    /**
     * @Route("/available_videos_lesson", name="available_videos_lesson", options={"expose"=true})
     * @Template()
     */
    public function showAvailableVideosAction()
    {
        try {
            $em = $this->getDoctrine()->getManager();
            $request = Request::createFromGlobals();
            $sv_video = $request->request->get('sv_video');

            $medias = $this->get("samba.videos.videos")->getMedias();

            $html = "<table>";
            foreach ($medias as $media) {
                $checked = ($media->id == $sv_video ? 'checked="checked"' : '');
                $html.= "<tr>";
                $html.= "<td>";
                $html.= "<input type='radio' ".$checked." name='sv_videos[]' value='{$media->id}' />";
                $html.= "</td>";
                $html.= "<td>";
                $html.= "<img src='{$media->thumbs[0]->url}' width='50px' />";
                $html.= "</td>";
                $html.= "<td>";
                $html.= $media->title;
                $html.= "</td>";
                $html.= "</tr>";
            }
            $html.= "</table>";

            $return = array("responseCode" => 200, "results" => $html);
        } catch (\Exception $e) {
            $return = array("responseCode" => 400, 'errorMessage' => $e->getTraceAsString());
        }

        return $this->returnJson($return);
    }

    /**
     * Displays a form to create a new Lesson entity.
     *
     * @Route("/new/{module}", name="backend_lesson_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction($module)
    {
        $this->verificationUserAccessControl("new_lesson");
        $em = $this->getDoctrine()->getManager();
        $module = $em->getRepository('AppBundle:Module')->find($module);

        if (!$module) {
            throw $this->createNotFoundException('Módulo não encontrado.');
        }
        $entity = new Lesson();
        $form   = $this->createCreateForm($entity);
        $medias = $this->get("samba.videos.videos")->getMedias();

        return array(
        'module' => $module,
        'entity' => $entity,
        'form'   => $form->createView(),
        );
    }

    /**
     * Lists all Lesson entities.
     *
     * @Route("/{module}", name="backend_lesson")
     * @Method("GET")
     * @Template()
     */
    public function indexAction(Request $request, $module)
    {
        $this->verificationUserAccessControl("index_lesson");
        $em = $this->getDoctrine()->getManager();

        $module = $em->getRepository('AppBundle:Module')->find($module);

        if (!$module) {
            throw $this->createNotFoundException('Módulo não encontrado.');
        }

        $entities = $em->getRepository("AppBundle:Lesson")->findBy(['module' => $module], ['position' => 'ASC']);

        return array(
            'module' => $module,
            'entities' => $entities,
        );
    }
    /**
     * Creates a new Lesson entity.
     *
     * @Route("/{module}", name="backend_lesson_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\Lesson:new.html.twig")
     */
    public function createAction(Request $request, $module = null)
    {
        $this->verificationUserAccessControl("new_lesson");
        $em = $this->getDoctrine()->getManager();
        $em->getConnection()->beginTransaction();
        $em->getConnection()->setAutoCommit(false);
        $entity = new Lesson();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);
        $module = $this->getModuleRepository()->find($module);
        if (!$module) {
            throw $this->createNotFoundException('Módulo não encontrado.');
        }

        if ($form->isValid()) {
            try {
                $position = $em->getRepository('AppBundle:Lesson')->getMaxLessonPositionByModule($module);
                $entity->setModule($module);
                $entity->setPosition($position+1);
                //$this->requestSambaVideosForm($request, $entity);

                $em->persist($entity);
                $em->flush();

                // Manager ScormFile
                if ($this->managerScormFile($entity, $request)) {
                    $em->getConnection()->commit();

                    $this->successCreateMessage($request);

                    return $this->redirect($this->generateUrl('backend_lesson', array(
                        'module' => $module->getId()
                    )));
                } else {
                    $this->errorMessage($request, 'Arquivo Scorm inválido!');
                    $em->getConnection()->rollback();
                    $em->close();
                    $entity->unlinkFiles();
                }
            } catch (\Exception $e) {
                $this->log($e);
                $this->errorMessage($request);
            }
        }

        return array(
            'module' => $module,
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Creates a form to create a Lesson entity.
     *
     * @param Lesson $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Lesson $entity)
    {
        $this->verificationUserAccessControl("new_lesson");
        $form = $this->createForm(new LessonType(), $entity, array(
            'action' => $this->generateUrl('backend_lesson_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to edit an existing Lesson entity.
     *
     * @Route("/{id}/edit", name="backend_lesson_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl("edit_lesson");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Lesson')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Lesson entity.');
        }

        $module = $entity->getModule();

        $editForm = $this->createEditForm($entity);

        return array(
            'module'      => $module,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a Lesson entity.
    *
    * @param Lesson $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Lesson $entity)
    {
        $this->verificationUserAccessControl("edit_lesson");
        $form = $this->createForm(new LessonType(), $entity, array(
            'action' => $this->generateUrl('backend_lesson_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing Lesson entity.
     *
     * @Route("/{id}", name="backend_lesson_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\Lesson:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("edit_lesson");
        $em = $this->getDoctrine()->getManager();
        $em->getConnection()->beginTransaction();
        $em->getConnection()->setAutoCommit(false);

        $entity = $em->getRepository('AppBundle:Lesson')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Lesson entity.');
        }

        $module = $entity->getModule();

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);
        $scormTemp = $entity->getScormFileTemp();
        if ($editForm->isValid()) {
            try {
                //$this->requestSambaVideosForm($request, $entity);
                $em->persist($entity);
                $em->flush();

                if ($this->managerScormFile($entity, $request, $scormTemp)) {
                    $em->getConnection()->commit();

                    $this->successMessage($request);

                    return $this->redirect($this->generateUrl('backend_lesson', array(
                        'module' => $module->getId()
                    )));
                } else {
                    $this->errorMessage($request, 'Arquivo Scorm inválido!');
                    $em->getConnection()->rollback();
                    $em->close();
                    $entity->unlinkFiles();
                }
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()
                ->getFlashBag()
                ->add('error', $e->getMessage());
            }
        }

        return array(
            'module'      => $module,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }
    /**
     * Deletes a Lesson entity.
     *
     * @Route("/{id}/delete", name="backend_lesson_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("remove_lesson");
        $em = $this->getDoctrine()->getManager();
        $lesson = $em->getRepository('AppBundle:Lesson')->find($id);

        if (!$lesson) {
            throw $this->createNotFoundException('Unable to find Lesson entity.');
        }

        $module = $lesson->getModule();

        $canRemove = true;
        $msgError = 'Não é possível remover essa aula devido o(s) seguinte(s) erro(s): <br/>';

        if (count($lesson->getComments())) {
            $msgError .= '- Possui '.count($lesson->getComments()).' comentários da aula. <br/>';
            $canRemove = false;
        }
        if (count($lesson->getInscriptionlessons())) {
            $msgError .= '- Possui '.count($lesson->getInscriptionlessons()).' inscrições. <br/>';
            $canRemove = false;
        }
        if (count($lesson->getLessonFiles())) {
            $msgError .= '- Possui '.count($lesson->getLessonFiles()).' arquivos associado(s) a aula selecionada. <br/>';
            $canRemove = false;
        }
        if (count($lesson->getScos())) {
            $msgError .= '- Possui '.count($lesson->getScos()).' scorms associado(s) a aula selecionada. <br/>';
            $canRemove = false;
        }

        try {
            if ($canRemove) {
                $em->remove($lesson);
                $em->flush();
                $lesson->unlinkFiles();
                $this->successMessage($request, 'Registro excluído com sucesso!');
            } else {
                $this->warningMessage($request, $msgError);
            }
        } catch (\Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_lesson', array('module' => $module->getId())));
    }

    private function managerScormFile($entity, $request, $scormTemp = null)
    {
        if ($entity->getScormFile()!=null) {
            $em = $this->getDoctrine()->getManager();
            $scos = $this->generateScos12FromScormArchive(Lesson::UPLOAD_PATH_LESSON_SCORM."/".$entity->getScormFile(), $request);
            // $this->log($scos);
            if (!$scos) {
                return false;
            }

            /*****************************************************************/
            /*                       Refatorar                               */
            /*****************************************************************/
            if ($scormTemp != null) {
                // Apagar os que tem o mesmo lesson_id
                $sco = $em->getRepository('AppBundle:Scorm12Sco')->findBy(array(
                    'lesson' => $entity
                ));

                if ($sco and is_array($sco)) {
                    foreach ($sco as $item) {
                        $em->remove($item);
                    }
                }
            }

            $scosPersist = array();
            // Corrigir para não duplicar

            // $this->log(json_encode($scos));

            if ($scos and is_array($scos)) {
                foreach ($scos as $item) {
                    //$this->log(gettype($item));
                    // $this->log($item->getTitle());

                    if (is_array($item)) {
                        $item = $item[0];
                    }

                    $sco = $em->getRepository('AppBundle:Scorm12Sco')->findBy(array(
                        'identifier' => $item->getIdentifier(),
                        'lesson' => $entity

                    ));
                    if (!$sco) {
                        array_push($scosPersist, $item);
                    }
                }
            }

            // $this->log('Verificar se está duplicando');
            // $this->log(count($scosPersist));

            /*****************************************************************/
            /*                       End                                     */
            /*****************************************************************/
            if (count($scosPersist) > 0) {
                $this->persistScos($entity, $scos);
            }
        }

        return true;
    }

    /**
     * Parses imsmanifest.xml file of a Scorm archive and
     * creates Scos defined in it.
     *
     * @param SplFileInfo $file
     *
     * @return array of Scorm resources
     */
    private function generateScos12FromScormArchive($file, $request)
    {
        $contents = '';
        $zip = new \ZipArchive();

        $zip->open($file);
        $stream = $zip->getStream("imsmanifest.xml");

        if (!$stream) {
            return false;
        }

        while (!feof($stream)) {
            $contents .= fread($stream, 2);
        }
        $dom = new \DOMDocument();
        try {
            if (!$dom->loadXML($contents)) {
                // $this->errorMessage($request, 'Não é possível carregar o arquivo "imsmanifest.xml". Verifique o arquivo antes de tentar novo upload.');
                throw $this->createNotFoundException('Não é possível carregar o arquivo "imsmanifest.xml"');
            }
        } catch (\Exception $e) {
            $this->errorMessage($request, 'Não é possível carregar o arquivo "imsmanifest.xml". Verifique o arquivo antes de tentar novo upload.');
        }
        
        //$this->log($dom->saveXML());

        $scormVersionElements = $dom->getElementsByTagName('schemaversion');

        if ($scormVersionElements->length > 0
            && $scormVersionElements->item(0)->textContent !== '1.2') {
            throw $this->createNotFoundException('A versão Scorm definida em "imsmanifest.xml" deve ser "1.2".');
        }
        // $this->log("[unimed]!!!!!");
        // $this->log($dom);
        $scos = $this->parseOrganizationsNode($dom);

        return $scos;
    }

    /**
     * Associates SCORM resource to SCOs and persists them.
     * As array $scos can also contain an array of scos
     * this method is call recursively when an element is an array.
     *
     * @param Scorm12Resource $scormResource
     * @param array $scos Array of Scorm12Sco
     */
    private function persistScos(
        Lesson $lesson,
        array $scos
    ) {
        $em = $this->getDoctrine()->getManager();
        foreach ($scos as $sco) {
            if (is_array($sco)) {
                $this->persistScos($lesson, $sco);
            } else {
                $sco->setLesson($lesson);
                $em->persist($sco);
            }
        }
        $em->flush();
    }

    /**
     * Looks for the organization to use
     *
     * @param \DOMDocument $dom
     * @return array of Scorm12Sco
     * @throws InvalidScormArchiveException If a default organization
     *         is defined and not found
     */
    public function parseOrganizationsNode(\DOMDocument $dom)
    {
        $organizationsList = $dom->getElementsByTagName('organizations');
        $resources = $dom->getElementsByTagName('resource');

        if ($organizationsList->length > 0) {
            $organizations = $organizationsList->item(0);
            $organization = $organizations->firstChild;

            $org = (!is_null($organizations->attributes)) ? $organizations->attributes : null ;
            $defaultOrganization = (!is_null($org->getNamedItem('default'))) ? $org->getNamedItem('default')->nodeValue : null ;

            // No default organization is defined
            if (is_null($defaultOrganization)) {
                while (!is_null($organization)
                    && $organization->nodeName !== 'organization') {
                    $organization = $organization->nextSibling;
                }

                if (is_null($organization)) {
                    return $this->parseResourceNodes($resources);
                }
            } // A default organization is defined
            // Look for it
            else {
                //$this->log('Entrei no else - 1');
                while (!is_null($organization)
                    && (
                        $organization->nodeName !== 'organization'
                        || is_null($organization->attributes->getNamedItem('identifier'))
                    /*|| $organization->attributes->getNamedItem('identifier')->nodeValue !== $defaultOrganization*/
                    )) {
                    $organization = $organization->nextSibling;
                }

                if (is_null($organization)) {
                    throw $this->createNotFoundException('A organização padrão definida em "imsmanifest.xml" não pode ser encontrada.');
                }
            }

            // $this->log('Iniciando parseItemNodes');
            return $this->parseItemNodes($organization, $resources);
        }
    }

    /**
     * Creates defined structure of SCOs
     *
     * @param \DOMNode $source
     * @param \DOMNodeList $resources
     * @return array of Scorm12Sco
     * @throws InvalidScormArchiveException
     */
    private function parseItemNodes(
        \DOMNode $source,
        \DOMNodeList $resources,
        Scorm12Sco $parentSco = null
    ) {
        $item = $source->firstChild;
        $scos = array();

        while (!is_null($item)) {
            if ($item->nodeName === 'item') {
                $sco = new Scorm12Sco();
                $scos[] = $sco;
                $sco->setScoParent($parentSco);
                $this->findAttrParams($sco, $item, $resources);
                $this->findNodeParams($sco, $item->firstChild);

                if ($sco->getIsBlock()) {
                    $scos[] = $this->parseItemNodes($item, $resources, $sco);
                }
            }
            $item = $item->nextSibling;
        }

        return $scos;
    }

    /**
     * Initializes parameters of the SCO defined in attributes of the node.
     * It also look for the associated resource if it is a SCO and not a block.
     *
     * @param Scorm12Sco $sco
     * @param \DOMNode $item
     * @param \DOMNodeList $resources
     * @throws InvalidScormArchiveException
     */
    private function findAttrParams(
        Scorm12Sco $sco,
        \DOMNode $item,
        \DOMNodeList $resources
    ) {
        $identifier = $item->attributes->getNamedItem('identifier');
        $isVisible = $item->attributes->getNamedItem('isvisible');
        $identifierRef = $item->attributes->getNamedItem('identifierref');
        $parameters = $item->attributes->getNamedItem('parameters');

        // throws an Exception if identifier is undefined
        if (is_null($identifier)) {
            throw $this->createNotFoundException('Um SCO definido em "imsmanifest.xml" não possui identificador');
        }
        $sco->setIdentifier($identifier->nodeValue);

        // visible is true by default
        if (!is_null($isVisible) && $isVisible === 'false') {
            $sco->setVisible(false);
        } else {
            $sco->setVisible(true);
        }

        // set parameters for SCO entry resource
        if (!is_null($parameters)) {
            $sco->setParameters($parameters->nodeValue);
        }

        // check if item is a block or a SCO. A block doesn't define identifierref
        if (is_null($identifierRef)) {
            $sco->setIsBlock(true);
        } else {
            $sco->setIsBlock(false);
            // retrieve entry URL
            $sco->setEntryUrl(
                $this->findEntryUrl($identifierRef->nodeValue, $resources)
            );
        }
    }

    /**
     * Searches for the resource with the given id and retrieve URL to its content.
     *
     * @param string $identifierref id of the resource associated to the SCO
     * @param \DOMNodeList $resources
     * @return string URL to the resource associated to the SCO
     * @throws InvalidScormArchiveException
     */
    public function findEntryUrl($identifierref, \DOMNodeList $resources)
    {
        foreach ($resources as $resource) {
            $identifier = $resource->attributes->getNamedItem('identifier');

            if (!is_null($identifier)) {
                $identifierValue = $identifier->nodeValue;

                if ($identifierValue === $identifierref) {
                    $href = $resource->attributes->getNamedItem('href');

                    if (is_null($href)) {
                        throw $this->createNotFoundException('Um SCO definido em "imsmanifest.xml" é anexado a um resource vazio');
                    }

                    return $href->nodeValue;
                }
            }
        }
        //throw new createNotFoundException('sco_without_resource_message');
    }


    /**
     * Initializes parameters of the SCO defined in children nodes
     *
     * @param Scorm12Sco $sco
     * @param \DOMNode $item
     */
    private function findNodeParams(Scorm12Sco $sco, \DOMNode $item)
    {
        while (!is_null($item)) {
            switch ($item->nodeName) {
                case 'title':
                    $sco->setTitle($item->nodeValue);
                    break;
                case 'adlcp:masteryscore':
                    $sco->setMasteryScore($item->nodeValue);
                    break;
                case 'adlcp:maxtimeallowed':
                    $sco->setMaxTimeAllowed($item->nodeValue);
                    break;
                case 'adlcp:timelimitaction':
                    $action = strtolower($item->nodeValue);

                    if ($action === 'exit,message'
                        || $action === 'exit,no message'
                        || $action === 'continue,message'
                        || $action === 'continue,no message') {
                        $sco->setTimeLimitAction($action);
                    }
                    break;
                case 'adlcp:datafromlms':
                    $sco->setLaunchData($item->nodeValue);
                    break;
                case 'adlcp:prerequisites':
                    $sco->setPrerequisites($item->nodeValue);
                    break;
            }
            $item = $item->nextSibling;
        }
    }
}
