<?php

namespace AppBundle\Controller\Backend;

use AppBundle\Controller\BaseController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Message;
use AppBundle\Form\MessageType;

/**
 * Message controller.
 *
 * @Route("/backend/message")
 */
class MessageController extends BaseController
{
    /**
     * @Route("/" , name="backend_message")
     * @Template("AppBundle:Backend\Message:index.html.twig")
     */
    public function indexAction()
    {
        $this->verificationUserAccessControl('index_message_topic');
        $em = $this->getDoctrine()->getManager();

        $entities = $em->getRepository('AppBundle:Message')->findAll();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $entities,
            $this->get('request')->query->get('page', 1),
            12
        );

        return array(
            'entities' => $entities,
        );
    }

    /**
     * @Route("/new/{topic_id}", name="backend_message_new")
     * @Template("AppBundle:Backend\Message:new.html.twig")
     */
    public function newAction(Request $request, $topic_id)
    {
        $this->verificationUserAccessControl('new_message_topic');
        $entity = new Message();
        $em = $this->getDoctrine()->getManager();

        $form = $this->createCreateForm($entity, $topic_id);
        $form->handleRequest($request);

        if ($request->isMethod('POST') || $request->isMethod('PUT')) {
            $topic = $em->getRepository('AppBundle:Topic')->find($topic_id);
            $entity->setTopic($topic);

            if (!$topic) {
                throw $this->createNotFoundException('Topico não encontrado.');
            }

            $em->persist($entity);
            $em->flush();

            $this->successMessage($request);

            return $this->redirect($this->generateUrl('backend_message_by_topic', array(
                'topic_id' => $topic_id
            )));
        }

        return array(
            'entity' => $entity,
            'form'   => $form->createView(),
            'topic_id' => $topic_id
        );
    }

    /**
     * @Route("/{id}/edit", name="backend_message_edit")
     * @Template("AppBundle:Backend\Message:edit.html.twig")
     */
    public function editAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('edit_message_topic');
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Message')->find($id);

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        $topic = $entity->getTopic();

        if ($request->isMethod('POST') || $request->isMethod('PUT')) {
            $em = $this->getDoctrine()->getManager();

            $em->persist($entity);
            $em->flush();

            $this->successMessage($request);

            return $this->redirect($this->generateUrl('backend_message_by_topic', array(
                'topic_id' => $topic->getId()
            )));
        }

        $editForm = $this->createEditForm($entity);

        return array(
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
            'topic_id'    => $topic->getId()
        );
    }

    /**
     * @Route("/{id}/delete", name="backend_message_delete")
     * @Template()
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_message_topic');
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Message')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Message entity.');
        }

        $topic = $entity->getTopic();

        $em->remove($entity);
        $em->flush();

        $this->successRemoveMessage($request);

        return $this->redirect($this->generateUrl('backend_message_by_topic', array(
            'topic_id' => $topic->getId()
        )));
    }

    /**
     * @Route("/topic/{topic_id}", name="backend_message_by_topic")
     * @Template("AppBundle:Backend\Message:index.html.twig")
     */
    public function messagesByTopicAction(Request $request, $topic_id)
    {
        $this->verificationUserAccessControl('index_message_topic');
        
        $em = $this->getDoctrine()->getManager();

        $topic = $em->getRepository('AppBundle:Topic')->find($topic_id);

        $entities = $em->getRepository('AppBundle:Message')->findBy(array(
            'topic' => $topic
        ));

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $entities,
            $this->get('request')->query->get('page', 1),
            12
        );

        return array(
            'topic' => $topic,
            'entities' => $entities,
        );
    }

    /**
     * Creates a form to create a Message entity.
     *
     * @param Message $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Message $entity, $topic_id)
    {
        $form = $this->createForm(new MessageType(), $entity, array(
            'action' => $this->generateUrl('backend_message_new', array(
                'topic_id' => $topic_id
            )),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
    * Creates a form to edit a Message entity.
    *
    * @param Message $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Message $entity)
    {
        $form = $this->createForm(new MessageType(), $entity, array(
            'action' => $this->generateUrl('backend_message_edit', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
}
