<?php

namespace AppBundle\Controller\Backend;

use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Module;
use AppBundle\Form\ModuleType;

/**
 * Module controller.
 *
 * @Route("/backend/module")
 */
class ModuleController extends BaseController
{

  /**
   * @Route("/module_sort_ajax_list", name="module_sort_ajax_list", options={"expose"=true})
   * @Template()
   */
    public function sortAjaxListAction()
    {
        try {
            $em = $this->getDoctrine()->getManager();
            $request = Request::createFromGlobals();
            $moduleOrder = $request->request->get('module_order');

            $listingCounter = 1;
            $moduleOrder = explode("&", $moduleOrder);
            foreach ($moduleOrder as $moduleId) {
                $moduleId = str_replace("module_order[]=", "", $moduleId);
                $module = $em->getRepository('AppBundle:Module')->find($moduleId);
                $module->setPosition($listingCounter);
                $em->persist($module);
                $em->flush();
                $listingCounter++;
            }

            $return = array("responseCode" => 200, "results" => $moduleOrder);
        } catch (\Exception $e) {
            $return = array("responseCode" => 400, 'errorMessage' => $e->getTraceAsString());
        }

        return $this->returnJson($return);
    }

    /**
     * Displays a form to create a new Module entity.
     *
     * @Route("/new/{course}", name="backend_module_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction($course)
    {
        $this->verificationUserAccessControl("new_module");
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->find($course);

        if (!$course) {
            throw $this->createNotFoundException('Curso não encontrado.');
        }
        $entity = new Module();
        $form   = $this->createCreateForm($entity);

        return array(
        'course' => $course,
        'entity' => $entity,
        'form'   => $form->createView(),
      );
    }

    /**
     * Lists all Module entities.
     *
     * @Route("/{course}", name="backend_module")
     * @Method("GET")
     * @Template()
     */
    public function indexAction(Request $request, $course)
    {
        $this->verificationUserAccessControl("index_module");
        $em = $this->getDoctrine()->getManager();

        $course = $em->getRepository('AppBundle:Course')->find($course);

        if (!$course) {
            throw $this->createNotFoundException('Curso não encontrado.');
        }

        $entities = $em->getRepository("AppBundle:Module")->findBy(['course' => $course], ['position' => 'ASC']);

        return array(
            'course' => $course,
            'entities' => $entities,
        );
    }
    /**
     * Creates a new Module entity.
     *
     * @Route("/{course}", name="backend_module_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\Module:new.html.twig")
     */
    public function createAction(Request $request, $course = null)
    {
        $this->verificationUserAccessControl("new_module");
        $em = $this->getDoctrine()->getManager();
        $entity = new Module();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);
        $course = $em->getRepository('AppBundle:Course')->find($course);
        if (!$course) {
            throw $this->createNotFoundException('Curso não encontrado.');
        }
        if ($form->isValid()) {
            try {
                $position = $em->getRepository('AppBundle:Module')->getMaxModulePositionByCourse($course);
                $entity->setCourse($course);
                $entity->setPosition($position+1);
                $em->persist($entity);
                $em->flush();
                $request->getSession()->getFlashBag()->add('success', 'Registro criado com sucesso!');
                return $this->redirect($this->generateUrl('backend_module', array('course' => $course->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()->getFlashBag()->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }
        return array(
            'course' => $course,
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Creates a form to create a Module entity.
     * @param Module $entity The entity
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Module $entity)
    {
        $this->verificationUserAccessControl("new_module");
        $form = $this->createForm(new ModuleType(), $entity, array(
            'action' => $this->generateUrl('backend_module_create'),
            'method' => 'POST',
        ));
        $form->add('submit', 'submit', array('label' => 'Criar'));
        return $form;
    }

    /**
     * Displays a form to edit an existing Module entity.
     * @Route("/{id}/edit", name="backend_module_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl("edit_module");
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Module')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Module entity.');
        }
        $course = $entity->getCourse();
        $editForm = $this->createEditForm($entity);
        return array(
            'course'      => $course,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a Module entity.
    * @param Module $entity The entity
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Module $entity)
    {
        $this->verificationUserAccessControl("edit_module");
        $form = $this->createForm(new ModuleType(), $entity, array(
            'action' => $this->generateUrl('backend_module_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));
        $form->add('submit', 'submit', array('label' => 'Atualizar'));
        return $form;
    }
    /**
     * Edits an existing Module entity.
     * @Route("/{id}", name="backend_module_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\Module:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("edit_module");
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Module')->find($id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Module entity.');
        }
        $course = $entity->getCourse();
        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);
        if ($editForm->isValid()) {
            try {
                $em->flush();
                $request->getSession()->getFlashBag()->add('success', 'Registro atualizado com sucesso!');
                return $this->redirect($this->generateUrl('backend_module', array('course' => $course->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()->getFlashBag()->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }
        return array(
            'course'      => $course,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }
    /**
     * Deletes a Module entity.
     *
     * @Route("/{id}/delete", name="backend_module_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("remove_module");
        $em = $this->getDoctrine()->getManager();
        $module = $em->getRepository('AppBundle:Module')->find($id);
        if (!$module) {
            throw $this->createNotFoundException('Unable to find Module entity.');
        }
        $course = $module->getCourse();

        $canRemove = true;
        $msgError = 'Não é possível remover esse módulo devido o(s) seguinte(s) erro(s): <br/>';

        // if ($course->getTotalLessons()) {
        //     $msgError .= '- Curso associado ao módulo selecionado, possui '.$course->getTotalLessons().' aula(s) associada(s). <br/>';
        //     $canRemove = false;
        // }
        // if ($course->getTotalEvaluations()) {
        //     $msgError .= '- Curso associado ao módulo selecionado, possui '.$course->getTotalEvaluations().' avaliações. <br/>';
        //     $canRemove = false;
        // }
        // if (count($course->getFeedbacks())) {
        //     $msgError .= '- Curso associado ao módulo selecionado, possui '.count($course->getFeedbacks()).' feedback dos usuarios. <br/>';
        //     $canRemove = false;
        // }
        // if (count($course->getSurveyanswers())) {
        //     $msgError .= '- Curso associado ao módulo selecionado, possui '.count($course->getSurveyanswers()).' pesquisa(s) de satisfação. <br/>';
        //     $canRemove = false;
        // }
        if (count($module->getLessons())) {
            $msgError .= '- Possui ' . count($module->getLessons()) . ' aula(s) associada ao módulo selecionado. <br/>';
            $canRemove = false;
        }
        if (count($module->getEvaluations())) {
            $msgError .= '- Possui ' . count($module->getEvaluations()) . ' avaliações de usuario(s). <br/>';
            $canRemove = false;
        }

        try {
            if ($canRemove) {
                $em->remove($module);
                $em->flush();
                $this->successMessage($request, 'Registro excluído com sucesso!');
            } else {
                $this->warningMessage($request, $msgError);
            }
        } catch (\Exception $e) {
            $this->log($e);
            $this->errorMessage($request, 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_module', array('course' => $course->getId())));
    }
}
