<?php

namespace AppBundle\Controller\Backend;

use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Question;
use AppBundle\Form\QuestionType;

/**
 * Question controller.
 *
 * @Route("/backend/question")
 */
class QuestionController extends BaseController
{


  /**
   * Displays a form to create a new Question entity.
   *
   * @Route("/new/{evaluation}", name="backend_question_new")
   * @Method("GET")
   * @Template()
   */
  public function newAction($evaluation)
  {
        $this->verificationUserAccessControl("new_question");
      $em = $this->getDoctrine()->getManager();
      $evaluation = $em->getRepository('AppBundle:Evaluation')->find($evaluation);

      if (!$evaluation) {
          throw $this->createNotFoundException('Avaliação não encontrada.');
      }
      $entity = new Question();
      $form   = $this->createCreateForm($entity);

      return array(
        'evaluation' => $evaluation,
        'entity' => $entity,
        'form'   => $form->createView(),
      );
  }

    /**
     * Lists all Questions entities.
     *
     * @Route("/{evaluation}", name="backend_question")
     * @Method("GET")
     * @Template()
     */
    public function indexAction(Request $request, $evaluation)
    {
        $this->verificationUserAccessControl("index_question");
        $em = $this->getDoctrine()->getManager();

        $evaluation = $em->getRepository('AppBundle:Evaluation')->find($evaluation);

        if (!$evaluation) {
            throw $this->createNotFoundException('Avaliação não encontrada.');
        }

        $entities = $em->getRepository("AppBundle:Question")->findBy(['evaluation' => $evaluation]);

        return array(
            'evaluation' => $evaluation,
            'entities' => $entities,
        );
    }
    /**
     * Creates a new Question entity.
     *
     * @Route("/{evaluation}", name="backend_question_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\Question:new.html.twig")
     */
    public function createAction(Request $request, $evaluation = null)
    {
        $this->verificationUserAccessControl("new_question");
        $em = $this->getDoctrine()->getManager();
        $entity = new Question();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);
        $evaluation = $this->getEvaluationRepository()->find($evaluation);
        if (!$evaluation) {
            throw $this->createNotFoundException('Avaliação não encontrada.');
        }

        if ($form->isValid()) {
            try{
              $entity->setEvaluation($evaluation);
              $em->persist($entity);
              $em->flush();

              $request->getSession()
                   ->getFlashBag()
                   ->add('success', 'Registro criado com sucesso!');

              return $this->redirect($this->generateUrl('backend_question', array('evaluation' => $evaluation->getId())));
            }
            catch(\Exception $e)
            {
              $this->log($e);
                $request->getSession()
                ->getFlashBag()
                ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'evaluation' => $evaluation,
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Creates a form to create a Question entity.
     *
     * @param Question $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Question $entity)
    {
        $this->verificationUserAccessControl("new_question");
        $form = $this->createForm(new QuestionType(), $entity, array(
            'action' => $this->generateUrl('backend_question_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to edit an existing Question entity.
     *
     * @Route("/{id}/edit", name="backend_question_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl("edit_question");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Question')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Question entity.');
        }

        $evaluation = $entity->getEvaluation();

        $editForm = $this->createEditForm($entity);

        return array(
            'evaluation'      => $evaluation,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a Question entity.
    *
    * @param Question $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Question $entity)
    {
        $this->verificationUserAccessControl("edit_question");
        $form = $this->createForm(new QuestionType(), $entity, array(
            'action' => $this->generateUrl('backend_question_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing Question entity.
     *
     * @Route("/{id}", name="backend_question_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\Question:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("edit_question");
        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:Question')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Question entity.');
        }

        $evaluation = $entity->getEvaluation();

        //Variáveis de controle para manter registros a serem excluídos
        $originalQuestionItemsAction = array();

        // Cria um array com questionItems já existentes
        foreach ($entity->getQuestionitems() as $questionItem) {
            $originalQuestionItemsAction[] = $questionItem;
        }

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
          try{
            //Tratamento para a persistência de serviços
            // Filtra o array $originalQuestionItemsAction para conter apenas itens que não existam mais
            foreach ($entity->getQuestionitems() as $questionItem) {
                foreach ($originalQuestionItemsAction as $key => $toDel) {
                    if ($toDel->getId() === $questionItem->getId()) {
                        unset($originalQuestionItemsAction[$key]);
                    }
                }
            }

            // Remove os dias que foram apagadas
            foreach ($originalQuestionItemsAction as $questionItem) {
                $em->remove($questionItem);
            }
            $em->flush();

            $request->getSession()
                 ->getFlashBag()
                 ->add('success', 'Registro atualizado com sucesso!');

            return $this->redirect($this->generateUrl('backend_question', array('evaluation' => $evaluation->getId())));
          }catch(\Exception $e)
          {
            $this->log($e);
              $request->getSession()
              ->getFlashBag()
              ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
          }

        }

        return array(
            'evaluation'      => $evaluation,
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }
    /**
     * Deletes a Question entity.
     *
     * @Route("/{id}/delete", name="backend_question_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl("remove_question");
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Question')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Question entity.');
        }

        $evaluation = $entity->getEvaluation();

        try{
          $em->remove($entity);
          $em->flush();
          $request->getSession()
               ->getFlashBag()
               ->add('success', 'Registro excluído com sucesso!');
        }catch(\Exception $e){
            $this->log($e);
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
        }

        return $this->redirect($this->generateUrl('backend_question', array('evaluation' => $evaluation->getId())));
    }

}
