<?php

namespace AppBundle\Controller\Backend;

use AppBundle\Controller\BaseController;
use AppBundle\Entity\ReportScript;
use AppBundle\Form\ReportScriptType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Symfony\Component\HttpFoundation\Request;

/**
 * ReportScript controller.
 *
 * @Route("/backend/report-script")
 */
class ReportScriptController extends BaseController
{
    /**
     * @Route("/" , name="backend_reportscript")
     * @Template("AppBundle:Backend\ReportScript:index.html.twig")
     */
    public function indexAction()
    {
        try {
            $this->verificationUserAccessControl('index_report');
            $em = $this->getDoctrine()->getManager();

            $entities = $em->getRepository('AppBundle:ReportScript')->findAll();

            return array(
                'entities' => $entities,
            );
        } catch (\Exception $e) {
            $this->log($e);
        }
    }

    /**
     * Creates a new ReportScript entity.
     *
     * @Route("/create", name="backend_reportscript_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\ReportScript:new.html.twig")
     */
    public function createAction(Request $request)
    {
        $this->verificationUserAccessControl('new_report');

        $entity = new ReportScript();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);
        if ($form->isValid()) {
            try {
                $em = $this->getDoctrine()->getManager();

                $em->persist($entity);
                $em->flush();

                $request->getSession()
                    ->getFlashBag()
                    ->add('success', 'Registro criado com sucesso!');

                return $this->redirect($this->generateUrl('backend_reportscript', array('id' => $entity->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity' => $entity,
            'form' => $form->createView(),
        );
    }

    /**
     * Creates a form to create a ReportScript entity.
     *
     * @param ReportScript $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(ReportScript $entity)
    {
        $form = $this->createForm(new ReportScriptType(), $entity, array(
            'action' => $this->generateUrl('backend_reportscript_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to create a new Category entity.
     *
     * @Route("/new", name="backend_reportscript_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction()
    {
        $this->verificationUserAccessControl('new_report');

        if ($this->get('security.authorization_checker')->isGranted('ROLE_SUPER_ADMIN')) {
            $entity = new ReportScript();
            $form = $this->createCreateForm($entity);

            return array(
                'entity' => $entity,
                'form' => $form->createView(),
            );
        } else {
            throw $this->createNotFoundException('Você não tem permissão para acessar esta tela.');
        }
    }

    /**
     * Displays a form to edit an existing Category entity.
     *
     * @Route("/{id}/edit", name="backend_reportscript_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl('edit_report');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:ReportScript')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find ReportScript entity.');
        }

        $editForm = $this->createEditForm($entity);
        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }

    /**
     * Creates a form to edit a ReportScript entity.
     *
     * @param ReportScript $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createEditForm(ReportScript $entity)
    {
        $form = $this->createForm(new ReportScriptType(), $entity, array(
            'action' => $this->generateUrl('backend_reportscript_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing Category entity.
     *
     * @Route("/{id}", name="backend_reportscript_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\ReportScript:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('edit_report');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:ReportScript')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find ReportScript entity.');
        }

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            try {
                $em->flush();

                $request->getSession()
                    ->getFlashBag()
                    ->add('success', 'Registro atualizado com sucesso!');

                return $this->redirect($this->generateUrl('backend_reportscript', array('id' => $id)));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }
    /**
     * Deletes a ReportScript entity.
     *
     * @Route("/{id}/delete", name="backend_reportscript_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_report');

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:ReportScript')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find ReportScript entity.');
        }

        $em->remove($entity);
        $em->flush();

        return $this->redirect($this->generateUrl('backend_reportscript'));
    }

    /**
     * @Route("/{id}/get-labels" , name="backend_reportscript_get_labels", options={"expose": true})
     * @Template("AppBundle:Backend\ReportScript:index.html.twig")
     */
    public function getLabelsAction($id)
    {
        $em = $this->getDoctrine()->getManager();
        $report = $em->getRepository('AppBundle:ReportScript')->find($id);

        return $this->returnJson(['labels' => $report->getLabels()]);
    }

    /**
     * @Route("/execute-query/{slug}" , name="backend_reportscript_executequery")
     * @Template("AppBundle:Backend\ReportScript:index.html.twig")
     */
    public function executeQueryAction(Request $request, $slug)
    {
        $this->verificationUserAccessControl('export_report');

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:ReportScript')->findOneBySlug($slug);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find ReportScript entity.');
        }

        if(!$entity->getIsRunning()){
            $script = $entity->getScript();
            foreach ($request->request->all() as $key => $value) {
                $script = str_replace('$' . $key, $value, $script);
            }
    
            $this->get('idb_queue')->push('appbundle.job.executereportscript', [
                'name' => $entity->getName(),
                'script' => $script,
                'slug' => $slug,
                'variables' => $request->request->all(),
                'email' => $this->getUser()->getEmail(),
                'url' => $this->getParameter('lms_url')
            ]);
    
            $request->getSession()
                ->getFlashBag()
                ->add('success', 'Relatório adicionado para fila de execução, em breve será enviado um e-mail com o resultado.');
    
            return $this->redirect($this->generateUrl('backend_reportscript'));
        }else{
            $request->getSession()
                ->getFlashBag()
                ->add('error', 'Este relatório já está sendo executado. Aguarde a confirmação para executar novamente.');
    
            return $this->redirect($this->generateUrl('backend_reportscript'));
        }

    }
}
