<?php

namespace AppBundle\Controller\Backend;

use AppBundle\Controller\BaseController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\Topic;
use AppBundle\Form\TopicType;

/**
 * Topic controller.
 *
 * @Route("/backend/topic")
 */
class TopicController extends BaseController
{
    /**
     * @Route("/", name="backend_topic")
     * @Template("AppBundle:Backend\Topic:index.html.twig")
     */
    public function indexAction()
    {
        $this->verificationUserAccessControl('index_topic');
        return array();
    }

    /**
     * @Route("/new/{theme_id}", name="backend_topic_new")
     * @Template("AppBundle:Backend\Topic:new.html.twig")
     */
    public function newAction(Request $request, $theme_id)
    {
        $this->verificationUserAccessControl('new_topic');
        $entity = new Topic();
        $em = $this->getDoctrine()->getManager();

        $form = $this->createCreateForm($entity, $theme_id);
        $form->handleRequest($request);

        if ($request->isMethod('POST')) {
            $theme = $em->getRepository('AppBundle:Theme')->find($theme_id);
            $entity->setTheme($theme);

            if (!$theme) {
                throw $this->createNotFoundException('Tema não encontrado.');
            }

            $em->persist($entity);
            $em->flush();

            $this->successMessage($request);

            return $this->redirect($this->generateUrl('backend_topic_by_theme', array(
                'theme_id' => $theme_id
            )));
        }

        return array(
            'entity' => $entity,
            'form'   => $form->createView(),
            'theme_id' => $theme_id
        );
    }

    /**
     * @Route("/{id}/edit", name="backend_topic_edit")
     * @Template("AppBundle:Backend\Topic:edit.html.twig")
     */
    public function editAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('edit_topic');
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Topic')->find($id);

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        $theme = $entity->getTheme();

        if ($request->isMethod('POST') || $request->isMethod('PUT')) {
            $em = $this->getDoctrine()->getManager();

            $em->persist($entity);
            $em->flush();

            $this->successMessage($request);

            return $this->redirect($this->generateUrl('backend_topic_by_theme', array(
                'theme_id' => $theme->getId()
            )));
        }

        $editForm = $this->createEditForm($entity);

        return array(
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
            'theme_id'    => $theme->getId()
        );
    }

    /**
     * @Route("/{id}/delete", name="backend_topic_delete")
     * @Template()
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_topic');
        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:Topic')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Topic entity.');
        }

        $theme = $entity->getTheme();

        $em->remove($entity);
        $em->flush();

        $this->successRemoveMessage($request);

        return $this->redirect($this->generateUrl('backend_topic_by_theme', array(
            'theme_id' => $theme->getId()
        )));
    }

    /**
     * Creates a form to create a Topic entity.
     *
     * @param Topic $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(Topic $entity, $theme_id)
    {
        $form = $this->createForm(new TopicType(), $entity, array(
            'action' => $this->generateUrl('backend_topic_new', array(
                'theme_id' => $theme_id
            )),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
    * Creates a form to edit a Topic entity.
    *
    * @param Topic $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(Topic $entity)
    {
        $form = $this->createForm(new TopicType(), $entity, array(
            'action' => $this->generateUrl('backend_topic_edit', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }

    /**
     * @Route("/theme/{theme_id}", name="backend_topic_by_theme")
     * @Template("AppBundle:Backend\Topic:index.html.twig")
     */
    public function topicsByThemeAction(Request $request, $theme_id)
    {
        $this->verificationUserAccessControl('index_topic');
        $em = $this->getDoctrine()->getManager();

        $theme = $em->getRepository('AppBundle:Theme')->find($theme_id);

        $topics = $em->getRepository('AppBundle:Topic')->findBy(array(
            'theme' => $theme
        ));

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $topics,
            $this->get('request')->query->get('page', 1),
            12
        );

        return array(
            'theme' => $theme,
            'entities' => $topics,
        );
    }
}
