<?php

namespace AppBundle\Controller\Backend;

use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use AppBundle\Form\UserGroupType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use AppBundle\Entity\UserGroup;

/**
 * @Route("backend/user-group")
 */
class UserGroupController extends BaseController
{

    /**
     * @Route("/" , name="backend_usergroup")
     * @Template("AppBundle:Backend\UserGroup:index.html.twig")
     */
    public function indexAction()
    {
        $this->verificationUserAccessControl('index_usergroup');
        $em = $this->getDoctrine()->getManager();

        $entities = $em->getRepository('AppBundle:UserGroup')->findAll();

        $paginator = $this->get('knp_paginator');
        $pagination = $paginator->paginate(
            $entities,
            $this->get('request')->query->get('page', 1),
            12
        );

        return array(
            'entities' => $entities,
        );
    }

    /**
     * Creates a new UserGroup entity.
     *
     * @Route("/create", name="backend_usergroup_create")
     * @Method("POST")
     * @Template("AppBundle:Backend\UserGroup:new.html.twig")
     */
    public function createAction(Request $request)
    {
        $this->verificationUserAccessControl('new_usergroup');

        $entity = new UserGroup();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);

        if ($form->isValid() && $this->validateForm($request, $entity)) {
            try {
                $em = $this->getDoctrine()->getManager();

                $em->persist($entity);
                $em->flush();

                $request->getSession()
                    ->getFlashBag()
                    ->add('success', 'Registro criado com sucesso!');

                return $this->redirect($this->generateUrl('backend_usergroup', array('id' => $entity->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity' => $entity,
            'form' => $form->createView(),
        );
    }
    
    function validateForm($request, $entity){
        $valid = true;
        if(empty($entity->getItemGroup())){
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'O campo Grupo é obrigatório!<br/>');
            $valid = false;
        }

        if(empty($entity->getItemKeys())){
            $request->getSession()
            ->getFlashBag()
            ->add('error', 'O campo Chaves é obrigatório!<br/>');
            $valid = false;
        }

        return $valid;
    }

    /**
     * Creates a form to create a UserGroup entity.
     *
     * @param UserGroup $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(UserGroup $entity)
    {
        $form = $this->createForm(new UserGroupType(), $entity, array(
            'action' => $this->generateUrl('backend_usergroup_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to create a new UserGroup entity.
     *
     * @Route("/new", name="backend_usergroup_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction()
    {
        $this->verificationUserAccessControl('new_usergroup');

        $entity = new UserGroup();
        $form = $this->createCreateForm($entity);

        return array(
            'entity' => $entity,
            'form' => $form->createView(),
        );
    }

    /**
     * Displays a form to edit an existing UserGroup entity.
     *
     * @Route("/{id}/edit", name="backend_usergroup_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl('edit_usergroup');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:UserGroup')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find UserGroup entity.');
        }

        $editForm = $this->createEditForm($entity);
        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a UserGroup entity.
    *
    * @param UserGroup $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(UserGroup $entity)
    {
        $form = $this->createForm(new UserGroupType(), $entity, array(
            'action' => $this->generateUrl('backend_usergroup_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing UserGroup entity.
     *
     * @Route("/{id}", name="backend_usergroup_update")
     * @Method("PUT")
     * @Template("AppBundle:Backend\UserGroup:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('edit_usergroup');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('AppBundle:UserGroup')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find UserGroup entity.');
        }

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid() && $this->validateForm($request, $entity)) {
            try {
                $em->flush();

                $request->getSession()
                    ->getFlashBag()
                    ->add('success', 'Registro atualizado com sucesso!');
                
                $em->getRepository('AppBundle:Client')->updateAccessGranted($entity->getActive(), $entity->getId());

                return $this->redirect($this->generateUrl('backend_usergroup', array('id' => $id)));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }
    /**
     * Deletes a UserGroup entity.
     *
     * @Route("/{id}/delete", name="backend_usergroup_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_usergroup');

        if ($this->get('security.authorization_checker')->isGranted('ROLE_SUPER_ADMIN')) {
            $this->verificationUserAccessControl('remove_usergroup');
            $em = $this->getDoctrine()->getManager();
            $userGroup = $em->getRepository('AppBundle:UserGroup')->find($id);

            if (!$userGroup) {
                throw $this->createNotFoundException('Unable to find UserGroup entity.');
            }

            $canRemove = true;
            $msgError = 'Não é possível remover esse Grupo devido o(s) seguinte(s) erro(s): <br/>';

            $users = $em->getRepository('UserBundle:User')->findByUsergroup($userGroup);
            if (count($users)) {
                $msgError .= '- Possui ' . count($users) . ' usuário(s) associado(s) a este grupo. <br/>';
                $canRemove = false;
            }
            if ($canRemove) {
                $em->remove($userGroup);
                $em->flush();

                $request->getSession()
                ->getFlashBag()
                ->add('success', 'Registro excluído com sucesso!');
            } else {
                $this->warningMessage($request, $msgError);
            }

            return $this->redirect($this->generateUrl('backend_usergroup'));
        } else {
            throw $this->createNotFoundException('Você não tem permissão para acessar esta tela.');
        }
    }

    /**
     * Deletes a UserGroup entity.
     *
     * @Route("/{id}/active", name="backend_usergroup_active")
     *
     */
    public function activeAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_usergroup');

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('AppBundle:UserGroup')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find UserGroup entity.');
        }

        if ($entity->getActive() == true) {
            $entity->setActive(false);
        } else {
            $entity->setActive(true);
        }
        $em->flush();

        return $this->redirect($this->generateUrl('backend_usergroup_index'));
    }
}
