<?php

namespace AppBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use AppBundle\Controller\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use AppBundle\Entity\Inscription;
use AppBundle\Entity\InscriptionStatus;
use AppBundle\Entity\Participant;
use AppBundle\Entity\ParticipantStatus;
use AppBundle\Entity\Payment;
use AppBundle\Entity\PaymentStatus;

/**
 * Course controller.
 * @Route("/matricula")
 */
class InscriptionController extends BaseController
{
    /**
     *
     * @Route("/{slug}/iniciar-curso", name="course_company_subscribe")
     */
    public function subscribeCompanyAction(Request $request, $slug)
    {
        $this->validUser();
        $course = $this->getCourseRepository()->findOneBySlug($slug);
        if (!$course) {
            throw $this->createNotFoundException('Curso não encontrado');
        }

        if ($course->getModality() == 'Presencial') {
            return $this->redirectToRoute('course_modality_show', array(
                'slug' => $course->getSlug()
            ));
        }

        $em   = $this->getDoctrine()->getManager();
        $user = $this->getUser();

        $userInscription = $em->getRepository('AppBundle:Inscription')->getLastInscriptionByUser($user, $course);
        if ($userInscription == null || $userInscription->getInscriptionStatus()->getId() == InscriptionStatus::CANCELADO) {
            //Verifica se existe vaga disponível
            $courseQuantVagas = $course->getQuantvagas();

            $inscriptionStatusAprovado = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO);
            //   $inscriptionOfCourse = $em->getRepository('AppBundle:Inscription')->findBy(array(
            //       'course' => $course,
            //       'inscription_status' => $inscriptionStatusAprovado,
            //   ));

            $inscriptionOfCourse = $em->getRepository('AppBundle:Inscription')->findTotalActiveInscriptions(
                $course,
                $inscriptionStatusAprovado
            );

            if ($courseQuantVagas != null && intval($courseQuantVagas) <= intval($inscriptionOfCourse)) {
                $request->getSession()
                   ->getFlashBag()
                   ->add('warning', 'Não é possível realizar este curso. A quantidade de vagas já foi preenchida!');
                return $this->redirectToRoute('courses_index');
            }

            $inscription = new Inscription();
            $inscription->setUser($user);
            $inscription->setCourse($course);

            if ($course->getDisponibleDays()!=null) {
                ///Instancia um novo objeto de matrícula
                $dtEnd = new \DateTime();
                if ($course->getDisponibleDays() > 730) {
                    $dtEnd->add(new \DateInterval('P730D'));
                } else {
                    $dtEnd->add(new \DateInterval('P'. $course->getDisponibleDays() .'D'));
                }

                $dtBegin = new \DateTime();
                $inscription->setDtBegin($dtBegin);
                $inscription->setDtEnd($dtEnd);
            }

            $inscription->setInscriptionStatus($em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
            $em->persist($inscription);
            $em->flush();

            $linkExternal = $course->getLinkTrailExternal();
            if (!empty($linkExternal)) {
                return $this->redirect($linkExternal);
            }

            return $this->redirectToRoute('course_start', array("id" => $course->getId()));
        } else {
            $linkExternal = $course->getLinkTrailExternal();
            if (!empty($linkExternal)) {
                return $this->redirect($linkExternal);
            }
            $inscriptonLesson = $em->getRepository('AppBundle:InscriptionLesson')->getMaxInscriptionLesson($userInscription);
            if ($inscriptonLesson==null) {
                return $this->redirectToRoute('course_start', array("id" => $course->getId()));
            }
            //Verifica se a aula possui scorm
            if ($inscriptonLesson->getLesson()->getScormFile()!=null) {
                $lesson = $inscriptonLesson->getLesson();
                $scos = $lesson->getScos();

                foreach ($scos as $sco) {
                    $scoTrackingVerify = $this->getScorm12ScoTrackingRepository()->findOneBy(array(
                        'user' => $user->getId(),
                        'sco' => $sco->getId()
                    ));
                    if ($scoTrackingVerify!=null && ($scoTrackingVerify->getLessonStatus() != "completed" && $scoTrackingVerify->getLessonStatus() != "COMPLETED")) {
                        return $this->redirectToRoute('lesson_show', array('courseId' => $course->getId(), 'id' => $inscriptonLesson->getLesson()->getId(), 'scoId' => $sco->getId()));
                    }
                }
            }

            return $this->redirectToRoute('lesson_show', array('courseId' => $course->getId(), 'id' => $inscriptonLesson->getLesson()->getId()));
        }
    }


    /**
     *
     * @Route("/{courseId}/presencial/inscription-waiting", name="inscription_course_waiting", options={"expose"=true})
     */
    public function inscriptionWaitingListAction(Request $request, $courseId)
    {
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->findOneById($courseId);
        $user = $this->getUser();

        if (!$this->checkAvailableParticipantCourse($course)) {
            $email = $request->request->get('user_email');
            $this->createParticipantWaiting($user, $course, $email);
            $this->successMessage($request, 'Inscrição realizada com sucesso.');
        }

        return $this->redirectToRoute('course_modality_show', array(
            'slug' => $course->getSlug()
        ));
    }

    /**
     *
     * @Route("/{courseId}/presencial/inscription-cancel", name="inscription_course_cancel", options={"expose"=true})
     */
    public function inscriptionCancelAction(Request $request, $courseId)
    {
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->findOneById($courseId);
        $user = $this->getUser();

        $participants = $em->getRepository('AppBundle:Participant')->findBy(array(
            'user' => $user,
            'course' => $course,
        ));

        if ($participants) {
            foreach ($participants as $participant) {
                $participantStatus = $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::CANCELADO);
                $participant->setParticipantStatus($participantStatus);
                $em->persist($participant);
            }
            $em->flush();
            $this->successMessage($request, 'Desistência realizada com sucesso.');
            $this->sendNotificationMailInscriptionInCourse($course);
        }

        return $this->redirectToRoute('course_modality_show', array(
            'slug' => $course->getSlug()
        ));
    }

    public function sendNotificationMailInscriptionInCourse($course)
    {
        $em = $this->getDoctrine()->getManager();
        $user = $this->getUser();
        $participant = $this->getFirstInscriptionWaiting($course);

        if ($participant) {
            $urlCourse = $this->generateUrl('course_modality_show', array('slug' => $course->getSlug()));

            $subject = 'Matricula Curso ' . $course->getName();
            $message = "Olá " . $participant->getUser()->getFirstName() . ",  <br />";
            $message .= "você estava na fila de espera para o curso " . $course->getName() . " .";
            $message .= "Ocorreu uma desistência e você foi incluído em nossa lista de participantes.";
            $message .= "Para mais informações, acesse http://lmssambatech.com.br" . $urlCourse;

            $email = $user->getEmail();
            

            if (!empty($participant->getEmail())) {
                $email = $participant->getEmail();
            }

            $participantStatus = $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::INSCRITO);
            $participant->setParticipantStatus($participantStatus);
            $em->persist($participant);
            $em->flush();

            $this->notificationUserEmail($email, $subject, $message);
        }
    }

    public function getFirstInscriptionWaiting($course)
    {
        $em = $this->getDoctrine()->getManager();
        $participant = $em->getRepository('AppBundle:Participant')->findOneBy(
            array(
                'course' => $course,
                'participant_status' => $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::ESPERA)
            ),
            array('id' => 'ASC')
        );

        return $participant;
    }

    /**
     *
     * @Route("/{slug}/modalidade/presencial", name="course_modality_show")
     */
    public function showModalityAction(Request $request, $slug)
    {
        $em = $this->getDoctrine()->getManager();
        $course = $em->getRepository('AppBundle:Course')->findOneBySlug($slug);
        $user = $this->getUser();
        $userCanInscription = true;

        $avaiableParticipantCourse = $this->checkAvailableParticipantCourse($course);
        $userParticipantWaiting = false;

        if (!$avaiableParticipantCourse) {
            $userParticipantWaiting = $this->checkParticipantWaitingInCourse($course, $user);
        }

        if ($request->getMethod() == 'POST') {
            // Fazer inscricão
            $this->createInscriptionApproved($user, $course);
            $this->successMessage($request, 'Inscrição realizada com sucesso.');
        }

        $participantUser = $em->getRepository('AppBundle:Participant')->findOneParticipantByUser($user, $course, true);

        return $this->render('AppBundle:Course:show.modality.html.twig', array(
            'course' => $course,
            'participant_user' => $participantUser,
            'avaiable_participant_course' => $avaiableParticipantCourse,
            'user_participant_waiting' => $userParticipantWaiting,
            'user_can_inscription' => $userCanInscription,
        ));
    }

    public function checkAvailableParticipantCourse($course)
    {
        $em = $this->getDoctrine()->getManager();
        $user = $this->getUser();

        $repoInscription = $em->getRepository('AppBundle:Participant');
        $totalParticipant = $repoInscription->getInscriptionsByCourseInDateCurrent($course);

        return $totalParticipant < $course->getQuantvagas();
    }

    public function checkParticipantWaitingInCourse($course, $user = false)
    {
        $em = $this->getDoctrine()->getManager();

        $filters = array(
            'course' => $course,
            'participant_status' => $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::ESPERA)
        );

        if ($user) {
            $filters['user'] = $user;
        }

        $inscription = $em->getRepository('AppBundle:Participant')->findOneBy($filters);

        return $inscription;
    }

    public function createParticipantWaiting($user, $course, $email)
    {
        $em = $this->getDoctrine()->getManager();
        $participant = $em->getRepository('AppBundle:Participant')->findOneBy(array(
            'user' => $user,
            'course' => $course,
        ));
        if (!$participant) {
            $participant = new Participant();
            $participant->setUser($user);
            $participant->setCourse($course);
        }
        $participant->setEmail($email);
        $participantStatus = $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::ESPERA);
        $participant->setParticipantStatus($participantStatus);

        $em->persist($participant);
        $em->flush();

        return $participant;
    }

    public function createInscriptionApproved($user, $course)
    {
        $em = $this->getDoctrine()->getManager();
        $participant = $em->getRepository('AppBundle:Participant')->findOneBy(array(
            'user' => $user,
            'course' => $course,
        ));
        if (!$participant) {
            $participant = new Participant();
            $participant->setUser($user);
            $participant->setCourse($course);
        }
        $participantStatus = $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::INSCRITO);
        $participant->setParticipantStatus($participantStatus);

        $em->persist($participant);
        $em->flush();

        return $participant;
    }

    public function changeInscriptionStatusForApproved($participant)
    {
        $participantStatus = $em->getReference('AppBundle:ParticipantStatus', ParticipantStatus::INSCRITO);
        $participant->setInscriptionStatus($participantStatus);

        return $participant;
    }

    public function teste($user)
    {
        $message = '';

        $swiftMessage = \Swift_Message::newInstance()
            ->setSubject("Unimed Curso Presencial")
            ->setFrom("universidadecorporativa@unimedbh.com.br")
            ->setTo($user->getEmail())
            ->setBody($message);
        
        $this->get('mailer')->send($swiftMessage);
    }
}
