<?php

namespace AppBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Validator\Constraints as Assert;
use AppBundle\Service\UploadService;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Mapping\Annotation as Gedmo;
use JMS\Serializer\Annotation\ExclusionPolicy;
use JMS\Serializer\Annotation\Expose;
use JMS\Serializer\Annotation\Type;

/**
* Class
*
* @ORM\Table(name="post")
* @ORM\Entity(repositoryClass="AppBundle\Repository\PostRepository")
* @ORM\HasLifecycleCallbacks()
*/
class Post
{

    /**
    * @var integer
    *
    * @ORM\Column(name="id", type="integer")
    * @ORM\Id
    * @ORM\GeneratedValue(strategy="AUTO")
    */
    private $id;

    /**
    * @var string
    *
    * @ORM\Column(name="title", type="string", length=255)
    */
    private $title;

    /**
    * @var string
    *
    * @ORM\Column(name="short_content", type="string", length=150)
    */
    private $short_content;

    /**
    * @var string
    *
    * @ORM\Column(name="content", type="text")
    */
    private $content;

    /**
    * @ORM\OneToMany(targetEntity="CommentPost", mappedBy="post", cascade={"all"})
    */
    protected $comments;

    /**
    * @var \DateTime
    *
    * @ORM\Column(name="dt_creation", type="datetime")
    * @Gedmo\Timestampable(on="create")
    */
    private $dtCreation;

    /**
    * @var \DateTime
    *
    * @ORM\Column(name="dt_update", type="datetime")
    * @Gedmo\Timestampable(on="update")
    */
    private $dtUpdate;

    /**
    * @var string
    *
    * @ORM\Column(name="image", type="string", length=255, nullable=true)
    */
    private $file;

    /**
    * @var string
    *
    * @ORM\OneToMany(targetEntity="LikePost", mappedBy="post", cascade={"all"})
    */
    private $likes;

    /**
    * @var string
    *
    * @ORM\Column(name="active", type="boolean")
    */
    private $active;

    /**
     * @var string
     *
     * @ORM\Column(name="slug", type="string", length=255)
     * @Gedmo\Slug(fields={"title"})
     */
    private $slug;

    /**
    * Constructor
    */
    public function __construct()
    {
        $this->comments = new \Doctrine\Common\Collections\ArrayCollection();
        $this->likes = new \Doctrine\Common\Collections\ArrayCollection();
    }

    /**
    * Get id
    *
    * @return integer
    */
    public function getId()
    {
        return $this->id;
    }

    /**
    * Set title
    *
    * @param string $title
    *
    * @return Post
    */
    public function setTitle($title)
    {
        $this->title = $title;

        return $this;
    }

    /**
    * Get title
    *
    * @return string
    */
    public function getTitle()
    {
        return $this->title;
    }

    /**
    * Set autor
    *
    * @param string $autor
    *
    * @return Post
    */
    public function setAutor($autor)
    {
        $this->autor = $autor;

        return $this;
    }

    /**
    * Get autor
    *
    * @return string
    */
    public function getAutor()
    {
        return $this->autor;
    }

    /**
    * Set short_content
    *
    * @param string $short_content
    *
    * @return Post
    */
    public function setShortContent($short_content)
    {
        $this->short_content = $short_content;

        return $this;
    }

    /**
    * Get short_content
    *
    * @return string
    */
    public function getShortContent()
    {
        return $this->short_content;
    }

    /**
    * Set content
    *
    * @param string $content
    *
    * @return Post
    */
    public function setContent($content)
    {
        $this->content = $content;

        return $this;
    }

    /**
    * Get content
    *
    * @return string
    */
    public function getContent()
    {
        return $this->content;
    }

    /**
    * Set dtCreation
    *
    * @param \DateTime $dtCreation
    *
    * @return Post
    */
    public function setDtCreation($dtCreation)
    {
        $this->dtCreation = $dtCreation;

        return $this;
    }

    /**
    * Get dtCreation
    *
    * @return \DateTime
    */
    public function getDtCreation()
    {
        return $this->dtCreation;
    }

    /**
    * Set dtUpdate
    *
    * @param \DateTime $dtUpdate
    *
    * @return Post
    */
    public function setDtUpdate($dtUpdate)
    {
        $this->dtUpdate = $dtUpdate;

        return $this;
    }

    /**
    * Get dtUpdate
    *
    * @return \DateTime
    */
    public function getDtUpdate()
    {
        return $this->dtUpdate;
    }

    
    /**
     * Set likes
     *
     * @param string $likes
     *
     * @return Post
     */
    public function setLikes($likes)
    {
        $this->likes = $likes;

        return $this;
    }

    /**
     * Get likes
     *
     * @return string
     */
    public function getLikes()
    {
        return $this->likes;
    }

    /**
    * Set active
    *
    * @param boolean $active
    *
    * @return Course
    */
    public function setActive($active)
    {
        $this->active = $active;

        return $this;
    }

    /**
    * Get active
    *
    * @return boolean
    */
    public function getActive()
    {
        return $this->active;
    }

    /**
    * Add comment
    *
    * @param \AppBundle\Entity\CommentPost $comment
    *
    * @return Post
    */
    public function addComment(\AppBundle\Entity\CommentPost $comment)
    {
        $this->comments[] = $comment;

        return $this;
    }

    /**
    * Remove comment
    *
    * @param \AppBundle\Entity\CommentPost $comment
    */
    public function removeComment(\AppBundle\Entity\CommentPost $comment)
    {
        $this->comments->removeElement($comment);
    }

    /**
    * Get comments
    *
    * @return \Doctrine\Common\Collections\Collection
    */
    public function getComments()
    {
        return $this->comments;
    }

    /**
     * Set file
     *
     * @param string $file
     *
     * @return Post
     */
    public function setFile($file)
    {
        $this->file = $file;

        return $this;
    }

    /**
     * Get file
     *
     * @return string
     */
    public function getFile()
    {
        return $this->file;
    }

    /**
     * Upload file
     */
    const UPLOAD_PATH_FILE = 'uploads/post/';

    /**
     * @Assert\Image(
     *     maxSize = "2048k",
     *     minWidth = "500",
     *     maxWidth = "520",
     *     minHeight = "330",
     *     maxHeight = "350",
     *     maxSizeMessage = "O tamanho do arquivo é muito grande. Escolha um arquivo de até 2MB",
     *     minWidthMessage = "A largura do arquivo é muito pequena. Escolha uma imagem próxima de 510 x 340",
     *     maxWidthMessage = "A largura do arquivo é muito grande. Escolha uma imagem próxima de 510 x 340",
     *     minHeightMessage = "A altura do arquivo é muito pequena. Escolha uma imagem próxima de 510 x 340",
     *     maxHeightMessage = "A altura do arquivo é muito grande. Escolha uma imagem próxima de 510 x 340"
     * )
     */
    private $fileTemp;

    /**
     * Sets fileTemp
     *
     * @param UploadedFile $fileTemp
     */
    public function setFileTemp(UploadedFile $fileTemp = null)
    {
        $this->fileTemp = $fileTemp;
    }

    /**
     * Get fileTemp
     *
     * @return UploadedFile
     */
    public function getFileTemp()
    {
        return $this->fileTemp;
    }

    /**
     * Unlink File
     */
    public function unlinkFiles()
    {
        if ($this->getFile() != null) {
            unlink(Post::UPLOAD_PATH_FILE ."/". $this->getFile());
        }
    }

    /**
     * Manages the copying of the file to the relevant place on the server
     */
    public function uploadFile()
    {

        //Upload de arquivo
        if($this->getFileTemp()!=null){
          //Se o diretorio não existir, cria
          if (!file_exists(Post::UPLOAD_PATH_FILE)) {
              mkdir(Post::UPLOAD_PATH_FILE, 0755, true);
          }
          if(
              ($this->getFileTemp() != $this->getFile())
              && (null !== $this->getFile())
          ){
              unlink(Post::UPLOAD_PATH_FILE ."/". $this->getFile());
          }

          // Generate a unique name for the file before saving it
          $fileName = md5(uniqid()).'.'.$this->getFileTemp()->guessExtension();

          UploadService::upload($this->getFileTemp(), Post::UPLOAD_PATH_FILE."/".$fileName);

          // set the path property to the filename where you've saved the file
          $this->file = $fileName;

          // clean up the file property as you won't need it anymore
          $this->setFileTemp(null);
      }

  }

    /**
     * Lifecycle callback to upload the file to the server
     * @ORM\PrePersist
     * @ORM\PreUpdate
     */
    public function lifecycleFileUpload() {
        $this->uploadFile();
    }

    /**
     * @ORM\PostLoad()
     */
    public function postLoad()
    {
        $this->dtUpdate = new \DateTime();
    }

    /**
     * Set slug
     *
     * @param string $slug
     * @return ServicesCategory
     */
    public function setSlug($slug)
    {
        $this->slug = $slug;

        return $this;
    }

    /**
     * Get slug
     *
     * @return string
     */
    public function getSlug()
    {
        return $this->slug;
    }
}
