<?php

namespace AppBundle\Repository;

use AppBundle\Entity\InscriptionStatus;
use AppBundle\Entity\Course;
use Doctrine\ORM\Query\ResultSetMapping;
use Doctrine\ORM\Query\Expr;

/**
 * InscriptionRepository
 *
 * This class was generated by the Doctrine ORM. Add your own custom
 * repository methods below.
 */
class InscriptionRepository extends \Doctrine\ORM\EntityRepository
{

    /**
     * Adiciona filtros para selecionar apenas registros ativos à query.
     * @param QueryBuilder $qb Query inicial.
     * @return QueryBuilder Query com filtros.
     */
    public function addActiveQuery(QueryBuilder $qb = null)
    {
        $em = $this->getEntityManager();

        if (is_null($qb)) {
            $qb = $em->createQueryBuilder();
        }

        $qb->select(array('i'))
            ->from('AppBundle:Inscription', 'i');

        return $qb;
    }

    /**
    * Busca todas as matriculas a partir dos dados informados
    * @param User $user Usuário corrente.
    * @param Course $course Curso a ser pesquisado.
    * @param InscriptionStatus $inscriptionstatus Status da matrícula.
    * @return array Array de registros.
    */
    public function findInscriptionByUser($user, $course = null, $inscriptionstatus = null)
    {
        $qb = $this->addActiveQuery();

        $qb->andWhere("i.user = :user")->setParameter("user", $user);

        if ($course!=null) {
            $qb->andWhere("i.course = :course")->setParameter("course", $course);
        }

        if ($inscriptionstatus!=null) {
            $qb->andWhere("i.inscription_status = :inscription_status");
            $qb->setParameter("inscription_status", $inscriptionstatus);
        }

        $qb->orderBy("i.id", "DESC");

        $query = $qb->getQuery();

        return $query->getResult();
    }

    public function getInscriptionsByCourseInDateCurrent($course, $inscriptionstatus)
    {
        $qb = $this->addActiveQuery();

        $qb->select('count(i.id) as total');

        $dateToday = new \DateTime();

        $qb->innerJoin('i.course', 'c');

        // $qb->andWhere("i.user = :user")->setParameter("user", $user);
        $qb->andWhere("i.course = :course")->setParameter("course", $course);

        $qb->andWhere("i.inscription_status = :inscription_status");
        $qb->setParameter("inscription_status", $inscriptionstatus);

        // $qb->andWhere('c.dtInitial < :dateToday');
        // $qb->andWhere('c.dtEnd > :dateToday');
        // $qb->setParameter("dateToday", $dateToday);

        $query = $qb->getQuery();

        // return $query->getResult();
        return $query->getSingleScalarResult();
    }

    public function getInscriptionsByCoursePresencial($course, $inscriptionStatus)
    {
        $qb = $this->addActiveQuery();

        $dateToday = new \DateTime();
        // $inscriptionStatusWaiting = 1;

        $qb->innerJoin('i.course', 'c');

        $qb->andWhere("i.course = :course")->setParameter("course", $course);

        $qb->andWhere("i.inscription_status = :inscription_status");
        $qb->setParameter("inscription_status", $inscriptionStatus);

        $qb->andWhere('c.dtInitial < :dateToday');
        $qb->andWhere('c.dtEnd > :dateToday');
        $qb->setParameter("dateToday", $dateToday);

        $qb->orderBy('i.dtCreation');

        $query = $qb->getQuery();

        return $query->getResult();
    }

    /**
    * Busca todas as matriculas a partir dos dados informados
    * @param User $user Usuário corrente.
    * @param Course $course Curso a ser pesquisado.
    * @param InscriptionStatus $inscriptionstatus Status da matrícula.
    * @return array Array de registros.
    */
    public function getLastInscriptionByUser($user, $course = null)
    {
        $qb = $this->addActiveQuery();

        $qb->andWhere("i.user = :user")->setParameter("user", $user);

        if ($course!=null) {
            $qb->andWhere("i.course = :course")->setParameter("course", $course);
            $data = new \DateTime();

            // $qb
        //   ->andWhere('(i.dtBegin <= :begin and i.dtEnd >= :end) or i.dtBegin is null ')
        //   ->setParameter('begin', $data->format('Y-m-d H:i:s'))
        //   ->setParameter('end', $data->format('Y-m-d H:i:s'));
        }

        $qb->orderBy("i.id", "DESC")->setMaxResults(1);

        return $qb->getQuery()->getOneOrNullResult();
    }

    /**
    * Busca todas as matriculas a partir dos dados informados
    * @param User $user Usuário corrente.
    * @param Course $course Curso a ser pesquisado.
    * @param InscriptionStatus $inscriptionstatus Status da matrícula.
    * @return array Array de registros.
    */
    public function getLastInscriptionByUserForApi($user, $course)
    {
        $qb = $this->addActiveQuery();

        $qb->andWhere("i.user = :user")->setParameter("user", $user);

        $qb->andWhere("i.course = :course")->setParameter("course", $course);

        $qb->orderBy("i.id", "DESC")->setMaxResults(1);

        return $qb->getQuery()->getOneOrNullResult();
    }

    /**
    * Verifica se o usuário possui uma matricula ativa no curso
    * @param User $user Usuário a ser pesquisado.
    * @param Course $course Curso a ser pesquisado.
    * @return boolean
    */
    public function verifyInscription($user, $course)
    {
        $em = $this->getEntityManager();
        $inscriptionStatus = $em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO);
        $inscriptions = $this->findInscriptionByUser($user, $course, $inscriptionStatus);
        return count($inscriptions) > 0 ? true : false;
    }

    /**
     * Retorna o total de inscritos no curso de acordo com os parâmetros informados.
     * @param Course $course
     * @param InscriptionStatus $status Status da inscrição a ser buscada
     * @return array Array de registros
     */
    public function getCountInscriptions(Course $course, InscriptionStatus $status = null)
    {
        $qb = $this->addActiveQuery();

        $qb->select('count(i.id) as total');

        $qb->andWhere('i.course = :course')
           ->setParameter('course', $course->getId());

        if ($status!=null) {
            $qb->andWhere('i.inscription_status = :status')
             ->setParameter('status', $status->getId());
        }

        $query = $qb->getQuery();

        return $query->getSingleScalarResult();
    }

    /**
    * Busca todas as matriculas para o relatório de acordo com o filtro
    * @param SearchReport $searchReport Entidade de busca para o report.
    * @return array Array de registros.
    */
    public function findForReport($searchReport)
    {
        $qb = $this->addActiveQuery();

        if ($searchReport->getBeginDate()!=null) {
            $qb->andWhere('i.dtCreation >= :dt_creation')
        ->setParameter('dt_creation', $searchReport->getBeginDate()->setTime(00, 00, 00));
        }

        if ($searchReport->getEndDate()!=null) {
            $qb->andWhere('i.dtCreation <= :dt_creation2')
        ->setParameter('dt_creation2', $searchReport->getEndDate()->setTime(23, 59, 59));
        }

        if ($searchReport->getCourse()!=null) {
            $qb->andWhere('i.course = :course')
        ->setParameter('course', $searchReport->getCourse());
        }

        if ($searchReport->getInscriptionStatus()!=null) {
            $qb->andWhere('i.inscription_status = :inscription_status')
        ->setParameter('inscription_status', $searchReport->getInscriptionStatus());
        }

        $query = $qb->getQuery();

        return $query->getResult();
    }

    /**
    * Busca todas as matriculas para o relatório de acordo com o filtro
    * @param SearchReport $searchReport Entidade de busca para o report.
    * @return array Array de registros.
    */
    public function findForFixUserEvaluation($course)
    {
        $qb = $this->addActiveQuery();
        // $qb->join("i.dtCreate","inscriptionlesson")
        $qb->andWhere("i.dtCreation >= :dtCreation")
        ->setParameter('dtCreation', "2017-08-21 00:00:00");

        $query = $qb->getQuery();

        return $query->getResult();
    }

    /**
     * Retorna o total de inscritos no curso de acordo com os parâmetros informados.
     * @param Course $course
     * @param InscriptionStatus $status Status da inscrição a ser buscada
     * @return array Array de registros
     */
    public function getCountInscriptionsFix(Course $course)
    {
        $qb = $this->addActiveQuery();

        $qb->select('count(i.id) as total');

        $qb->andWhere('i.course = :course')
           ->setParameter('course', $course->getId());

        $qb->andWhere("i.dtCreation >= :dtCreation")
        ->setParameter('dtCreation', "2017-08-21 00:00:00");

        // $qb->join("i.inscriptionlessons","inscriptionlesson")
        // ->andWhere("inscriptionlesson.finished is not null");

        $query = $qb->getQuery();

        return $query->getSingleScalarResult();
    }

    /**
     * Busca todos os pedidos a serem finalizados.
     * @return array Pedidos a serem finalizados.
     */
    public function findOpenedInscriptions()
    {
        $now = new \DateTime("now");

        $rsm = new ResultSetMapping;
        $rsm->addEntityResult('AppBundle:Inscription', 'i');
        $rsm->addFieldResult('i', 'id', 'id');
        $rsm->addJoinedEntityResult('AppBundle:Course', 'c', 'i', 'course');
        $rsm->addFieldResult('c', 'course_id', 'id');
        $rsm->addFieldResult('c', 'course_slug', 'slug');
        $rsm->addFieldResult('c', 'course_name', 'name');
        $rsm->addJoinedEntityResult('UserBundle:User', 'u', 'i', 'user');
        $rsm->addFieldResult('u', 'user_id', 'id');
        $rsm->addFieldResult('u', 'user_email', 'email');

        $query = $this->_em->createNativeQuery('select i.id, c.id as course_id, c.slug as course_slug,
c.name as course_name, u.id as user_id, u.email as user_email
from inscription i
inner join course c on i.course_id = c.id
inner join user u on i.user_id = u.id
inner join payment p on p.inscription_id = i.id
where i.inscriptionstatus_id = :inscriptionstatus_waiting
and TIMESTAMPDIFF(MINUTE, i.dt_creation, :now) >= 15
and gateway_code is null
order by i.dt_creation', $rsm);
        $query->setParameter('inscriptionstatus_waiting', InscriptionStatus::AGUARDANDO_PAGAMENTO);
        $query->setParameter('now', $now);

        $inscriptions = $query->getResult();

        return $inscriptions;
    }

    /**
    * Busca todas as matriculas a partir dos dados informados
    * @param String $mat_colaborador Usuário corrente - unimed.
    * @param String $codEvn Curso a ser pesquisado - unimed.
    * @return array Array de registros.
    */
    public function getInscriptionByUserEvent($codEvn=null, $mat_colaborador=null, $nota_frequencia = false, $limit = null)
    {
        $qb = $this->addActiveQuery();
        $qb->join("i.user", "user")
      ->leftJoin('AppBundle:Client', 'client', \Doctrine\ORM\Query\Expr\Join::WITH, 'user.id=client.id')
      ->join("i.course", "course");

        if ($mat_colaborador!=null) {
            $qb->andWhere('client.mat_colaborador = :mat_colaborador')
        ->setParameter('mat_colaborador', $mat_colaborador);
        }

        if ($codEvn!=null) {
            $qb->andWhere('course.codeevent = :codeevent')
        ->setParameter('codeevent', $codEvn);
        }

        if ($nota_frequencia) {
            $qb->join("user.userEvaluations", "userEvaluations")
            ->andWhere('userEvaluations.status is null');
        }

        if ($limit != null) {
            $qb->setMaxResults($limit);
        }

        $query = $qb->getQuery();

        return $query->getResult();
    }

    /**
    * Busca todas as matriculas a partir dos dados informados
    * @param User $superior Usuário
    * @return array Array de registros.
    */
    public function getInscriptionBySuperior($superior, $query = null, $paginator=true, $count=false, $counter=null)
    {
        $em = $this->getEntityManager();
        $qb = $em->createQueryBuilder();
        if ($count) {
            $qb->select('count(i.id)')
            ->from('AppBundle:Inscription', 'i');
        // $qb = $em->createQuery('SELECT COUNT(i) FROM AppBundle:Inscription i');
        } else {
            // $qb = $em->createQueryBuilder();
            $qb->select(array('i'))
            ->from('AppBundle:Inscription', 'i');
        }
        // $qb = $this->addActiveQuery();
        $qb->join("i.user", "user")
            ->join("i.inscription_status", "inscription_status")
            ->leftJoin('AppBundle:Client', 'client', \Doctrine\ORM\Query\Expr\Join::WITH, 'user.id=client.id')
            ->join("i.course", "course");


        $qb->andWhere('client.superior = :superior')
        ->andWhere("inscription_status.id = :inscription_status")
        ->setParameter('superior', $superior)
        ->setParameter("inscription_status", InscriptionStatus::APROVADO);
        $qb->orderBy("user.first_name");

        if ($query!=null) {
            $qb->andWhere(
                $qb->expr()->like($qb->expr()->lower('course.title'), ':query'). " or ".
                $qb->expr()->like($qb->expr()->lower('course.name'), ':query'). " or ".
                $qb->expr()->like($qb->expr()->lower('course.description'), ':query')." or ".
                $qb->expr()->like($qb->expr()->lower('client.first_name'), ':query')." or ".
                $qb->expr()->like($qb->expr()->lower('client.last_name'), ':query')
            )->setParameter(':query', "%" . strtolower($query) . "%");
        }

        if ($count) {
            $qb->getQuery()->getSingleScalarResult();
        } else {
            $query = $qb->getQuery();
            $query->setHint('knp_paginator.count', $counter);
            if ($paginator == true) {
                return $query;
            } else {
                return $query->getResult();
            }
        }
    }

    /**
    * Busca todas as matriculas a partir dos dados informados
    * @param User $superior Usuário
    * @return array Array de registros.
    */
    public function getInscriptionByUserActive($query)
    {
        $qb = $this->addActiveQuery();
        $qb->join("i.user", "user")
            ->join("i.inscription_status", "inscription_status")
            ->leftJoin('AppBundle:Client', 'client', \Doctrine\ORM\Query\Expr\Join::WITH, 'user.id=client.id')
            ->join("i.course", "course")

        ->andWhere("inscription_status.id = :inscription_status")
        ->setParameter("inscription_status", InscriptionStatus::APROVADO);
        $qb->orderBy("user.first_name");

        if ($query!=null) {
            $qb->andWhere(
                $qb->expr()->like($qb->expr()->lower('course.name'), ':query'). " or ".
                $qb->expr()->like($qb->expr()->lower('course.description'), ':query')." or ".
                $qb->expr()->like($qb->expr()->lower('client.first_name'), ':query')." or ".
                $qb->expr()->like($qb->expr()->lower('client.last_name'), ':query')
            )->setParameter(':query', "%" . strtolower($query) . "%");
        }

        return $qb;
    }

    public function findTotalActiveInscriptions($course, $inscriptionStatusAprovado)
    {
        $qb = $this->addActiveQuery();
        $qb->select('count(i.id) as totalActiveInscriptions');

        $qb->andWhere("i.course = :course")->setParameter("course", $course);
        $qb->andWhere("i.inscription_status = :inscription_status")->setParameter("inscription_status", $inscriptionStatusAprovado);

        return $qb->getQuery()->getSingleScalarResult();
    }

    /**
   * @param array $get
   * @param bool $flag
   * @return array|\Doctrine\ORM\Query
   */
    public function ajaxTable(array $get, $course = null, $presential = false)
    {
        /* Indexed column (used for fast and accurate table cardinality) */
        $alias = 'i';
        /* DB table to use */
        $tableObjectName = 'AppBundle:Inscription';
        /**
         * Set to default
         */
        if (!isset($get['columns']) || empty($get['columns'])) {
            $get['columns'] = array('id');
        }
        $aColumns = array();
        foreach ($get['columns'] as $value) {
            if (strpos($value, '.') !== false) {
                $aColumns[] = $value;
            } else {
                $aColumns[] = $alias .'.'. $value;
            }
        }
        $cb = $this->getEntityManager()
        ->getRepository($tableObjectName)
        ->createQueryBuilder($alias);
        //->select(str_replace(" , ", " ", implode(", ", $aColumns)));
        if (isset($get['iDisplayStart']) && $get['iDisplayLength'] != '-1') {
            $cb->setFirstResult((int)$get['iDisplayStart'])
            ->setMaxResults((int)$get['iDisplayLength']);
        }
        $cb->join($alias.".user", "user");
        $cb->join($alias.".course", "course");
        $cb->leftJoin('AppBundle:Client', 'client', \Doctrine\ORM\Query\Expr\Join::WITH, 'user.id=client.id');
        /*
         * Ordering
         */
        // if ( isset( $get['iSortCol_0'] ) ){
        //   for ( $i=0 ; $i<intval( $get['iSortingCols'] ) ; $i++ ){
        //     if ( $get[ 'bSortable_'.intval($get['iSortCol_'.$i]) ] == "true" ){
        //       $cb->orderBy($aColumns[ (int)$get['iSortCol_'.$i] ], $get['sSortDir_'.$i]);
        //     }
        //   }
        // }
        /*
           * Filtering
           * NOTE this does not match the built-in DataTables filtering which does it
           * word by word on any field. It's possible to do here, but concerned about efficiency
           * on very large tables, and MySQL's regex functionality is very limited
           */
        if (isset($get['sSearch']) && $get['sSearch'] != '') {
            $aLike = array();
            for ($i=0 ; $i<count($aColumns) ; $i++) {
                if (isset($get['bSearchable_'.$i]) && $get['bSearchable_'.$i] == "true") {
                    $aLike[] = $cb->expr()->like($aColumns[$i], '\'%'. $get['sSearch'] .'%\'');
                }
            }
            if (count($aLike) > 0) {
                $cb->andWhere(new Expr\Orx($aLike));
            } else {
                unset($aLike);
            }
        }

        if ($course!=null) {
            $cb->andWhere("i.course = :course")->setParameter("course", $course);
        }
        if ($presential) {
            $cb->andWhere("course.modality = :presencial")->setParameter("presencial", Course::MODALITY_PRESENCIAL);
            $cb->andWhere("i.inscription_status = :inscription_status")->setParameter("inscription_status", InscriptionStatus::AGUARDANDO_PAGAMENTO);
        }

        /*
         * SQL queries
         * Get data to display
         */
        $query = $cb->getQuery();
        return $query->getResult();
    }

    /**
     * @return int
     */
    public function getCount($course=null, $presential=false)
    {
        if ($course) {
            $query = $this->getEntityManager()
            ->createQuery('SELECT COUNT(i) FROM AppBundle:Inscription i join i.course course where course.id = ?1')
            ->setMaxResults(1)
            ->setParameter(1, $course->getId());
            $aResultTotal = $query->getResult();
        } else {
            if ($presential) {
                $query = $this->getEntityManager()
                ->createQuery('SELECT COUNT(i) FROM AppBundle:Inscription i join i.course course where course.modality = ?1 and i.inscription_status = ?2')
                ->setMaxResults(1)
                ->setParameter(1, Course::MODALITY_PRESENCIAL)
                ->setParameter(2, InscriptionStatus::AGUARDANDO_PAGAMENTO);
                
                $aResultTotal = $query->getResult();
            } else {
                $aResultTotal = $this->getEntityManager()
                ->createQuery('SELECT COUNT(i) FROM AppBundle:Inscription i')
                ->setMaxResults(1)
                ->getResult();
            }
        }

        
        return $aResultTotal[0][1];
    }
}
