<?php

namespace PagSeguroBundle\Service;

use AppBundle\Entity\Payment;
use AppBundle\Entity\InscriptionStatus;
use PagSeguroBundle\Entity\GatewayReturn;
use Doctrine\ORM\EntityManager;
use PagSeguroBundle\Entity\PagSeguroPaymentRequest;
use PagSeguroBundle\Entity\PagSeguroShippingType;
use PagSeguroBundle\Entity\PagSeguroAccountCredentials;
use PagSeguroBundle\Config\PagSeguroConfig;
use PagSeguroBundle\Service\PagSeguroNotificationService;
use PagSeguroBundle\Exception\PagSeguroServiceException;


class PagSeguroManager {

    protected $container;
    protected $em;
    protected $pagseguro_environment;
    protected $pagseguro_account;
    protected $pagseguro_token;
    protected $pagseguro_checkouturl;
    protected $app_enviroment_url;
    protected $logger;
    protected $mailer;

    public function __construct($container, EntityManager $em, $logger, $mailer, $pagseguro_environment = null, $pagseguro_account = null, $pagseguro_token = null, $pagseguro_checkouturl = null, $app_enviroment_url = null) {
        $this->container = $container;
        $this->em = $em;
        $this->pagseguro_environment = $pagseguro_environment;
        $this->pagseguro_account = $pagseguro_account;
        $this->pagseguro_token = $pagseguro_token;
        $this->pagseguro_checkouturl = $pagseguro_checkouturl;
        $this->app_enviroment_url = $app_enviroment_url;
        $this->logger = $logger;
        $this->mailer = $mailer;
    }

    /**
     * Através do pedido gerado pelo site é gerado um novo pedido para a compra com o pagseguro
     *
     * @param Payment $payment Registro de pagamento
     * @param Inscription $inscription Registro de matrícula
     * @param User $user Usuário logado
     * @return array $resultArray Resultado do processamento do cartão de crédito, informando se o mesmo foi processado ou não.
     */
    public function generateOrder($payment, $inscription, $user) {

        // Variável de retorno
        $gatewayReturn = new GatewayReturn();
        $gatewayReturn->setSuccess(false);

        $paymentRequest = new PagSeguroPaymentRequest();

        // Set the currency
        $paymentRequest->setCurrency("BRL");

        // Add an item for this payment request
        $paymentRequest->addItem('0001', $inscription->getCourse()->getName(), 1, $payment->getValue());

        // Set a reference code for this payment request. It is useful to identify this payment
        // in future notifications.
        $paymentRequest->setReference($payment->getId());

        // Set your customer information.
        $paymentRequest->setSender(
            $user->getFullName(),
            $user->getEmail()
        );

        // Set the url used by PagSeguro to redirect user after checkout process ends
        $paymentRequest->setRedirectUrl($this->app_enviroment_url."/pagamento/sucesso?payment_id=".$payment->getId());

        try {
            $this->logger->error("Confirma pagamento através do pagseguro");
            // seller authentication
            $credentials = new PagSeguroAccountCredentials($this->pagseguro_account, $this->pagseguro_token, $this->pagseguro_environment);
            // Seta como success o retorno
            $gatewayReturn->setSuccess(true);
            // Register this payment request in PagSeguro to obtain the payment URL to redirect your customer.
            $url = $paymentRequest->register($credentials);
            $gatewayReturn->setUrl($url);

        } catch (PagSeguroServiceException $e) {
            $this->logger->error("Ocorreu um erro!");
            die($e->getMessage());
        }
        return $gatewayReturn;
    }

    /**
    * Processa os dados recebidos através da notificação automatica
    * @param String $notificationCode Código de notificação recebido
    */
    function transactionNotification($notificationCode)
    {
        //Preenche as credenciais para acesso ao PagSeguro
        $credentials = new PagSeguroAccountCredentials($this->pagseguro_account,
          $this->pagseguro_token,
          $this->pagseguro_environment);

        try {
            //Checa a transação a partir da notificação recebida
            $transaction = PagSeguroNotificationService::checkTransaction($credentials, $notificationCode);

            if($transaction!=null){
              $this->logger->error("[PAGSEGURO] NOTIFICACAO - ".$transaction->toString());
              //Obtém o status da transação
              $transactionStatus = $transaction->getStatus()->getValue();

              //Busca o pagamento através do código
              $this->logger->error("[PAGSEGURO] NOTIFICACAO - BUSCANDO PAYMENT DE CODIGO ".$transaction->getCode());
              $payment = $this->em->getRepository('AppBundle:Payment')->findOneByGatewayCode($transaction->getCode());
              if($payment!=null){
                $paymentStatus = $this->em->getRepository('AppBundle:PaymentStatus')->find($transactionStatus);
                if($paymentStatus!=null){
                  $payment->setPaymentStatus($this->em->getReference('AppBundle:PaymentStatus', $transactionStatus));
                }
              }else{
                //Se o pagamento não for encontrado pelo código, busca através do id de referência
                $this->logger->error("[PAGSEGURO] NOTIFICACAO - CODIGO NAO ENCONTRADO. BUSCANDO PELO ID DE REFERENCIA ".$transaction->getReference());
                $payment = $this->em->getRepository('AppBundle:Payment')->findOneById($transaction->getReference());
                if($payment!=null){
                  $paymentStatus = $this->em->getRepository('AppBundle:PaymentStatus')->find($transactionStatus);
                  if($paymentStatus!=null){
                    $payment->setPaymentStatus($this->em->getReference('AppBundle:PaymentStatus', $transactionStatus));
                  }
                  $payment->setGatewayCode($transaction->getCode());
                }else{
                  $this->logger->error("[PAGSEGURO] NOTIFICACAO - PAGAMENTO NAO ENCONTRADO");
                  return false;
                }
              }
              $this->em->persist($payment);
              $this->em->flush();

              $this->logger->error("[PAGSEGURO] NOTIFICACAO - STATUS DE TRANSACAO $transactionStatus");
              $inscription = $payment->getInscription();
              //Aguardando pagamento / Em análise / Em disputa
              if($transactionStatus == 1 || $transactionStatus == 2 || $transactionStatus == 5){
                $inscription->setInscriptionStatus($this->em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::AGUARDANDO_PAGAMENTO));
              }else if($transactionStatus == 3 || $transactionStatus == 4){
                //Paga / Disponível
                $inscription->setInscriptionStatus($this->em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::APROVADO));
                $message = $this->createMessage($inscription, true);
                $this->sendEmail($inscription->getUser()->getEmail(), "Matrícula Confirmada", $message);
              }else if($transactionStatus == 6 || $transactionStatus == 7){
                //Devolvida / Cancelada
                $inscription->setInscriptionStatus($this->em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::CANCELADO));
                $message = $this->createMessage($inscription, false, true);
                $this->sendEmail($inscription->getUser()->getEmail(), "Matrícula Cancelada", $message);
              }else{
                //STATUS NAO MAPEADO
                $inscription->setInscriptionStatus($this->em->getReference('AppBundle:InscriptionStatus', InscriptionStatus::AGUARDANDO_PAGAMENTO));
              }
              $this->em->persist($inscription);
              $this->em->flush();

              return true;
            }else{
              //Notificação inválida
              return false;
            }
        } catch (PagSeguroServiceException $e) {
            die($e->getMessage());
        }
    }

    private function createMessage($entity, $confirmacao = false, $cancelamento = false){
      $message = '<div style="width: 100%; background: #004c83; padding: 10px 0;">';
      $message .= '<img src="http://enlab.com.br/bundles/app/img/frontend/logo.png" alt="Enlab" style="margin: 0 auto;display:block;"/>';
      $message .= '</div>';
      if($confirmacao){
        $message .= '<div style="width: 100%; background: #f3f3f3; padding: 30px 0; text-align:center;">';
        $message .= '    <h1 style="color: #191919; font-weight:500; font-size: 24px;margin: 20px 0;;">Confirma&ccedil;&atilde;o de Pagamento</h1>';
        $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Ol&aacute;, seu pagamento foi confirmado com sucesso!</p>';
        $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Acesse agora mesmo o curso '.$entity->getCourse().':</p>';
        $message .= '    <p><a href="http://www.enlab.com.br/curso/'.$entity->getCourse()->getSlug().'"><i style="display: inline-block; padding: 6px 12px; margin-bottom: 0;text-align: center; white-space: nowrap; vertical-align: middle;cursor: pointer;font-style:normal;border-radius: 4px;background: #004c83; border: 1px solid #004c83; color: #fff; font-size: 16px; font-weight: 700; transition: 0.3s; width: 145px; height: 40px; line-height: 2.668571;cursor: pointer">QUERO ASSISTIR</i></a>';
        $message .= '</div>';
      }
      if($cancelamento){
        $message .= '<div style="width: 100%; background: #f3f3f3; padding: 30px 0; text-align:center;">';
        $message .= '    <h1 style="color: #191919; font-weight:500; font-size: 24px;margin: 20px 0;;">Matr&iacute;cula Cancelada</h1>';
        $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Ol&aacute;, seu pagamento n&atilde;o foi aprovado e por isto a sua matr&iacute;cula foi cancelada</p>';
        $message .= '    <p style="color: #666666; font-weight: 400; font-size:16px;margin: 20px 0;">Voc&ecirc; pode realizar uma nova matr&iacute;cula, clicando no bot&atilde;o abaixo.:</p>';
        $message .= '    <p><a href="http://www.enlab.com.br/curso/'.$entity->getCourse()->getSlug().'"><i style="display: inline-block; padding: 6px 12px; margin-bottom: 0;text-align: center; white-space: nowrap; vertical-align: middle;cursor: pointer;font-style:normal;border-radius: 4px;background: #004c83; border: 1px solid #004c83; color: #fff; font-size: 16px; font-weight: 700; transition: 0.3s; width: 145px; height: 40px; line-height: 2.668571;cursor: pointer">NOVA MATR&Iacute;CULA</i></a>';
        $message .= '</div>';
      }
      return $message;
    }

    private function sendEmail($email, $subject, $message){
      $mailMessage = \Swift_Message::newInstance()
              ->setSubject("[Enlab] - ".$subject)
              ->setFrom("no-reply@enlab.com.br")
              ->setTo($email)
              ->setContentType("text/html")
              ->setBody($message);

      $this->mailer->send($mailMessage);
    }

}
