<?php

namespace UserBundle\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use UserBundle\Entity\User;
use UserBundle\Form\UserBackendType;
use AppBundle\Controller\BaseController;

/**
 * User controller.
 *
 * @Route("/backend/user")
 */
class UserController extends BaseController
{

    /**
     * Lists all User entities.
     *
     * @Route("/", name="backend_user")
     * @Method("GET")
     * @Template()
     */
    public function indexAction()
    {
        // if (!$this->get('security.authorization_checker')->isGranted('ROLE_SUPER_ADMIN')) {
        //     throw $this->createNotFoundException('Você não tem permissão para acessar esta tela.');
        // }

        $this->verificationUserAccessControl('index_user');

        $em = $this->getDoctrine()->getManager();

        $entities = $em->getRepository('UserBundle:User')->findOnlyUser('user');

        return array(
            'entities' => $entities,
        );
    }
    /**
     * Creates a new User entity.
     *
     * @Route("/", name="backend_user_create")
     * @Method("POST")
     * @Template("UserBundle:User:new.html.twig")
     */
    public function createAction(Request $request)
    {
        $this->verificationUserAccessControl('new_user');

        $entity = new User();
        $form = $this->createCreateForm($entity);
        $form->handleRequest($request);

        if ($form->isValid()) {
            if (!$this->validUsernameEmail($entity)) {
                $request->getSession()
                        ->getFlashBag()
                        ->add('error', 'Email já cadastrado, tente novamente!');
                return $this->redirectToRoute('backend_user_new');
            }

            try {
                $em = $this->getDoctrine()->getManager();

                $email = $form->get('email')->getData();

                // Set entity fields custom
                $this->requestCustomForm($request, $entity);

                $entity->setUsername($email);
                $entity->setEnabled(true);
                $entity->setAccessGranted(true);
                $entity->setRoles(array("ROLE_SUPER_ADMIN"));

                $em->persist($entity);
                $em->flush();

                $request->getSession()
                            ->getFlashBag()
                            ->add('success', 'Registro criado com sucesso!');

                return $this->redirect($this->generateUrl('backend_user', array('id' => $entity->getId())));
            } catch (\Exception $e) {
                $this->log($e);
                $request->getSession()
                            ->getFlashBag()
                            ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
            }
        }

        return array(
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Creates a form to create a User entity.
     *
     * @param User $entity The entity
     *
     * @return \Symfony\Component\Form\Form The form
     */
    private function createCreateForm(User $entity)
    {
        $form = $this->createForm(new UserBackendType(), $entity, array(
            'action' => $this->generateUrl('backend_user_create'),
            'method' => 'POST',
        ));

        $form->add('submit', 'submit', array('label' => 'Create'));

        return $form;
    }

    /**
     * Displays a form to create a new User entity.
     *
     * @Route("/new", name="backend_user_new")
     * @Method("GET")
     * @Template()
     */
    public function newAction()
    {
        $this->verificationUserAccessControl('new_user');

        $entity = new User();
        $form   = $this->createCreateForm($entity);

        return array(
            'entity' => $entity,
            'form'   => $form->createView(),
        );
    }

    /**
     * Displays a form to edit an existing User entity.
     *
     * @Route("/{id}/edit", name="backend_user_edit")
     * @Method("GET")
     * @Template()
     */
    public function editAction($id)
    {
        $this->verificationUserAccessControl('edit_user');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UserBundle:User')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find User entity.');
        }

        $editForm = $this->createEditForm($entity);
        return array(
            'entity'      => $entity,
            'edit_form'   => $editForm->createView(),
        );
    }

    /**
    * Creates a form to edit a User entity.
    *
    * @param User $entity The entity
    *
    * @return \Symfony\Component\Form\Form The form
    */
    private function createEditForm(User $entity)
    {
        $form = $this->createForm(new UserBackendType(), $entity, array(
            'action' => $this->generateUrl('backend_user_update', array('id' => $entity->getId())),
            'method' => 'PUT',
        ));

        $form->add('submit', 'submit', array('label' => 'Update'));

        return $form;
    }
    /**
     * Edits an existing User entity.
     *
     * @Route("/{id}", name="backend_user_update")
     * @Method("PUT")
     * @Template("UserBundle:User:edit.html.twig")
     */
    public function updateAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('edit_user');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UserBundle:User')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find User entity.');
        }

        $editForm = $this->createEditForm($entity);
        $editForm->handleRequest($request);

        if ($editForm->isValid()) {
            if ($this->validUsernameEmail($entity)) {
                try {
                    // Set entity fields custom
                    $this->requestCustomForm($request, $entity);
                    $entity->setUsername($entity->getEmail());

                    $em->flush();

                    return $this->redirect($this->generateUrl('backend_user', array('id' => $id)));
                } catch (\Exception $e) {
                    $this->log($e);
                    $request->getSession()
                        ->getFlashBag()
                        ->add('error', 'Ocorreu algum erro inesperado. Tente novamente mais tarde!');
                }
            } else {
                $request->getSession()
                    ->getFlashBag()
                    ->add('error', 'Email já cadastrado, tente novamente!');
            }
        }

        return array(
            'entity' => $entity,
            'edit_form' => $editForm->createView(),
        );
    }
    /**
     * Deletes a User entity.
     *
     * @Route("/{id}/delete", name="backend_user_delete")
     *
     */
    public function deleteAction(Request $request, $id)
    {
        $this->verificationUserAccessControl('remove_user');

        $em = $this->getDoctrine()->getManager();
        $entity = $em->getRepository('UserBundle:User')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find User entity.');
        }

        $em->remove($entity);
        $em->flush();

        return $this->redirect($this->generateUrl('backend_user'));
    }

    /**
    * Atualiza o Status do Usuários para ATIVO
    *
    * @Route("/?id={id}", name="backend_user_status")
    */
    public function activateStatusAction($id)
    {
        $this->verificationUserAccessControl('disable_user');

        $em = $this->getDoctrine()->getManager();

        $entity = $em->getRepository('UserBundle:User')->find($id);

        if (!$entity) {
            throw $this->createNotFoundException('Unable to find User entity.');
        }

        $condicao = $entity->getStatus();
        if ($condicao == "Habilitado") {
            $entity->setEnabled(false);
        } else {
            $entity->setEnabled(true);
        }

        $em->persist($entity);
        $em->flush();

        return $this->redirect($this->generateUrl('backend_user', array('id' => $id)));
    }

    /**
     * Set entity fields custom
     * @param Request $request
     * @param Entity $entity
     */
    public function requestCustomForm($request, $entity)
    {
        $em = $this->getDoctrine()->getManager();
        $userCustomArray = $request->request->get('user_custom');

        if (!empty($userCustomArray['plainPassword'])) {
            $encoder = $this->container->get('security.encoder_factory')->getEncoder($entity);
            $new_pwd_encoded = $encoder->encodePassword($userCustomArray['plainPassword'], $entity->getSalt());
            $entity->setPassword($new_pwd_encoded);
        }
    }

    /**
     * Valida se o username e email são válidos
     * @param User $entity
     * @return boolean $flag
     */
    public function validUsernameEmail($entity)
    {
        $em = $this->getDoctrine()->getManager();

        $flag = $em->getRepository('UserBundle:User')->getUserByNameAndUsername($entity);

        return $flag;
    }
}
