<?php

namespace UserBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use FOS\UserBundle\Model\User as BaseUser;
use FOS\UserBundle\Model\UserInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Validator\Constraints as Assert;
use AppBundle\Service\UploadService;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @ORM\HasLifecycleCallbacks
 * @ORM\Table(name="user")
 * @ORM\InheritanceType("JOINED")
 * @ORM\DiscriminatorColumn(name="discr", type="string")
 * @ORM\DiscriminatorMap({"user" = "User", "client" = "AppBundle\Entity\Client"})
 * @ORM\Entity(repositoryClass="UserBundle\Repository\UserRepository")
 * @ORM\AttributeOverrides({
 *  @ORM\AttributeOverride(
 *      name="salt",
 *      column=@ORM\Column(name="salt", type="string", nullable=true)
 *      ),
 *      @ORM\AttributeOverride(name="email", column=@ORM\Column(type="string", name="email", length=255, unique=false, nullable=true)),
 *      @ORM\AttributeOverride(name="emailCanonical", column=@ORM\Column(type="string", name="email_canonical", length=255, unique=false, nullable=true)),
 *      @ORM\AttributeOverride(name="username", column=@ORM\Column(type="string", name="username", length=180, unique=false, nullable=true)),
 *      @ORM\AttributeOverride(name="usernameCanonical", column=@ORM\Column(type="string", name="username_canonical", length=180, unique=false, nullable=true))
 *  })
 */
class User extends BaseUser implements UserInterface
{
    public function __construct()
    {
        parent::__construct();
        $this->inscriptions = new ArrayCollection();
        $this->inscriptionsClass = new ArrayCollection();
        $this->subordinates = new ArrayCollection();
        $this->permissions = new ArrayCollection();
    }

    /**
     * __toString method
     */
    public function __toString()
    {
        return $this->getFullName();
    }

    /**
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    /**
     * @var string
     *
     * @ORM\Column(name="first_name", type="string", length=100, nullable=true)
     */
    private $first_name;

    /**
     * @var string
     *
     * @ORM\Column(name="last_name", type="string", length=100, nullable=true)
     */
    private $last_name;

    /**
     * @var string
     *
     * @ORM\Column(name="celphone", type="string", length=255, nullable=true)
     */
    private $celphone;

    /**
     * @var string
     *
     * @ORM\Column(name="phone", type="string", length=255, nullable=true)
     */
    private $phone;

    /**
     * @var string
     *
     * @ORM\Column(name="uid", type="string", length=255, nullable=true)
     */
    private $uid;

    /**
     * @var string
     *
     * @ORM\Column(name="photo", type="text", nullable=true)
     */
    private $photo;

    /**
     * @var UserStatus
     *
     * @ORM\ManyToOne(targetEntity="AppBundle\Entity\UserStatus")
     * @ORM\JoinColumn(name="user_status_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private $userStatus;

    /**
     * @var JobRole
     *
     * @ORM\ManyToOne(targetEntity="AppBundle\Entity\JobRole")
     * @ORM\JoinColumn(name="job_role_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private $jobRole;

    /**
     * @var Management
     *
     * @ORM\ManyToOne(targetEntity="AppBundle\Entity\Management")
     * @ORM\JoinColumn(name="management_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private $management;

    /**
     * @var Sector
     *
     * @ORM\ManyToOne(targetEntity="AppBundle\Entity\Sector")
     * @ORM\JoinColumn(name="sector_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private $sector;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dt_creation", type="datetime", nullable=true)
     * @Gedmo\Timestampable(on="create")
     */
    private $dtCreation;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dt_update", type="datetime", nullable=true)
     * @Gedmo\Timestampable(on="update")
     */
    private $dtUpdate;

    /**
     * @ORM\OneToMany(targetEntity="AppBundle\Entity\Inscription", mappedBy="user", cascade={"all"})
     */
    protected $inscriptions;

    /**
     * @ORM\OneToMany(targetEntity="AppBundle\Entity\InscriptionClass", mappedBy="user", cascade={"all"})
     */
    protected $inscriptionsClass;

    /**
     * @ORM\OneToMany(targetEntity="AppBundle\Entity\UserEvaluation", mappedBy="user", cascade={"all"})
     */
    protected $userEvaluations;

    /**
     * @var User
     *
     * @ORM\ManyToOne(targetEntity="UserBundle\Entity\User", inversedBy="subordinates")
     * @ORM\JoinColumn(name="superior_id", referencedColumnName="id", nullable=true)
     */
    private $superior;

    /**
     * @ORM\OneToMany(targetEntity="UserBundle\Entity\User", mappedBy="superior")
     */
    protected $subordinates;

    /**
     * @ORM\OneToMany(targetEntity="AppBundle\Entity\UserAccessControl", mappedBy="user", cascade={"all"}, orphanRemoval=true)
     */
    protected $permissions;

    /**
     * @var UserGroup
     *
     * @ORM\ManyToOne(targetEntity="AppBundle\Entity\UserGroup")
     * @ORM\JoinColumn(name="usergroup_id", referencedColumnName="id", nullable=true, onDelete="SET NULL")
     */
    private $usergroup;

    /**
     * @var string
     *
     * @ORM\Column(name="access_granted", type="boolean")
     */
    private $accessGranted;

    /**
     * Retorna o nome completo.
     * @return string Nome completo.
     */
    public function getFullName()
    {
        return $this->getFirstName() . " " . $this->getLastName();
    }

    /**
     * Set id
     *
     * @param integer $id
     */
    public function setId($id)
    {
        $this->id = $id;
    }

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set first_name
     *
     * @param string $first_name
     * @return User
     */
    public function setFirstName($first_name)
    {
        $this->first_name = $first_name;

        return $this;
    }

    /**
     * Get first_name
     *
     * @return string
     */
    public function getFirstName()
    {
        return $this->first_name;
    }

    /**
     * Set last_name
     *
     * @param string $last_name
     * @return User
     */
    public function setLastName($last_name)
    {
        $this->last_name = $last_name;

        return $this;
    }

    /**
     * Get last_name
     *
     * @return string
     */
    public function getLastName()
    {
        return $this->last_name;
    }

    /**
     * Set celphone
     *
     * @param string $celphone
     * @return User
     */
    public function setCelphone($celphone)
    {
        $this->celphone = $celphone;

        return $this;
    }

    /**
     * Get celphone
     *
     * @return string
     */
    public function getCelphone()
    {
        return $this->celphone;
    }

    /**
     * Set phone
     *
     * @param string $phone
     * @return User
     */
    public function setPhone($phone)
    {
        $this->phone = $phone;

        return $this;
    }

    /**
     * Get phone
     *
     * @return string
     */
    public function getPhone()
    {
        return $this->phone;
    }

    /**
     * Set uid
     *
     * @param string $uid
     * @return User
     */
    public function setUid($uid)
    {
        $this->uid = $uid;

        return $this;
    }

    /**
     * Get uid
     *
     * @return string
     */
    public function getUid()
    {
        return $this->uid;
    }

    /**
     * Get enabled
     *
     * Retorna o Status do Usuário(habilidado,desabilitado)
     *
     */
    public function getStatus()
    {
        $status = $this->enabled;

        if ($status == 1) {
            $status = "Habilitado";
        } else {
            $status = "Desabilitado";
        }

        return $status;
    }

    /**
     * Set photo
     *
     * @param string $photo
     * @return User
     */
    public function setPhoto($photo)
    {
        $this->photo = $photo;

        return $this;
    }

    /**
     * Get photo
     *
     * @return string
     */
    public function getPhoto()
    {
        return $this->photo;
    }

    /**
     * Set userStatus
     *
     * @param UserStatu $userStatus
     * @return Class
     */
    public function setUserStatus($userStatus = null)
    {
        $this->userStatus = $userStatus;

        return $this;
    }

    /**
     * Get userStatus
     *
     * @return Class
     */
    public function getUserStatus()
    {
        return $this->userStatus;
    }

    /**
     * Set jobRole
     *
     * @param JobRole $jobRole
     * @return Class
     */
    public function setJobRole($jobRole = null)
    {
        $this->jobRole = $jobRole;

        return $this;
    }

    /**
     * Get jobRole
     *
     * @return Class
     */
    public function getJobRole()
    {
        return $this->jobRole;
    }

    /**
     * Set management
     *
     * @param Management $management
     * @return Class
     */
    public function setManagement($management = null)
    {
        $this->management = $management;

        return $this;
    }

    /**
     * Get management
     *
     * @return Class
     */
    public function getManagement()
    {
        return $this->management;
    }

    /**
     * Set sector
     *
     * @param Sector $sector
     * @return Class
     */
    public function setSector($sector = null)
    {
        $this->sector = $sector;

        return $this;
    }

    /**
     * Get sector
     *
     * @return Class
     */
    public function getSector()
    {
        return $this->sector;
    }

    /**
     * Set dtCreation
     *
     * @param \DateTime $dtCreation
     *
     * @return Payment
     */
    public function setDtCreation($dtCreation)
    {
        $this->dtCreation = $dtCreation;

        return $this;
    }

    /**
     * Get dtCreation
     *
     * @return \DateTime
     */
    public function getDtCreation()
    {
        return $this->dtCreation;
    }

    /**
     * Set dtUpdate
     *
     * @param \DateTime $dtUpdate
     *
     * @return Payment
     */
    public function setDtUpdate($dtUpdate)
    {
        $this->dtUpdate = $dtUpdate;

        return $this;
    }

    /**
     * Get dtUpdate
     *
     * @return \DateTime
     */
    public function getDtUpdate()
    {
        return $this->dtUpdate;
    }


    /**
     * Upload image
     */

    // Constante com o caminho para salvar a imagem screenshot
    const UPLOAD_PATH_USER_PHOTO = 'uploads/user/photo';

    // 512000  bytes / 500 kbytes
    // 1048576 bytes / 1024 kbytes
    // 2097152 bytes / 2048 kbytes

    /**
     * @Assert\File(
     *     maxSize = "3072k",
     *     maxSizeMessage = "O tamanho da imagem é muito grande ({{ size }} {{ suffix }}), escolha uma imagem de até {{ limit }} {{ suffix }}",
     *     mimeTypes = {"image/jpeg", "image/gif", "image/png"},
     *     mimeTypesMessage = "Formato de arquivo inválido. Formatos permitidos: .gif, .jpeg e .png"
     * )
     */
    private $photoTemp;

    /**
     * Sets photoTemp
     *
     * @param UploadedFile $photoTemp
     */
    public function setPhotoTemp(UploadedFile $photoTemp = null)
    {
        $this->photoTemp = $photoTemp;
    }

    /**
     * Get photoTemp
     *
     * @return UploadedFile
     */
    public function getPhotoTemp()
    {
        return $this->photoTemp;
    }

    /**
     * Unlink Photo (Apagar foto quando excluir objeto)
     */
    public function unlinkImages()
    {
        if ($this->getPhoto() != null) {
            unlink(User::UPLOAD_PATH_USER_PHOTO . "/" . $this->getPhoto());
        }
    }

    public function uploadImage()
    {

        //Upload de foto de usuário
        if ($this->getPhotoTemp() != null) {
            //Se o diretorio não existir, cria
            if (!file_exists(User::UPLOAD_PATH_USER_PHOTO)) {
                mkdir(User::UPLOAD_PATH_USER_PHOTO, 0755, true);
            }
            if (($this->getPhotoTemp() != $this->getPhoto())
                && (null !== $this->getPhoto())) {
                unlink(User::UPLOAD_PATH_USER_PHOTO . "/" . $this->getPhoto());
            }

            // Gera um nome único para o arquivo
            $fileName = md5(uniqid()) . '.' . $this->getPhotoTemp()->guessExtension();

            UploadService::compress($this->getPhotoTemp(), User::UPLOAD_PATH_USER_PHOTO . "/" . $fileName, 100);

            $this->photo = $fileName;
            $this->setPhotoTemp(null);
        }
    }

    /**
     * Lifecycle callback to upload the file to the server
     * @ORM\PrePersist
     * @ORM\PreUpdate
     */
    public function lifecycleFileUpload()
    {
        $this->uploadImage();
    }

    /**
     * @ORM\PostLoad()
     */
    public function postLoad()
    {
        $this->dtUpdate = new \DateTime();
    }

    /**
     *
     * @return type
     */
    public function getInscriptions()
    {
        return $this->inscriptions;
    }

    /**
     *
     */
    public function setInscriptions($inscriptions)
    {
        $this->inscriptions = $inscriptions;
    }

    /**
     *
     * @return type
     */
    public function getInscriptionsClass()
    {
        return $this->inscriptionsClass;
    }

    /**
     *
     */
    public function setInscriptionsClass($inscriptionsClass)
    {
        $this->inscriptionsClass = $inscriptionsClass;
    }

    /**
     *
     * @return type
     */
    public function getUserEvaluations()
    {
        return $this->userEvaluations;
    }

    /**
     *
     */
    public function setUserEvaluations($userEvaluations)
    {
        $this->userEvaluations = $userEvaluations;
    }

    /**
     *
     * @return type
     */
    public function getSubordinates()
    {
        return $this->subordinates;
    }

    /**
     *
     */
    public function setSubordinates($subordinates)
    {
        $this->subordinates = $subordinates;
    }

    public function verifyInscription($course)
    {
        // $inscription = $this->getInscriptions()->last();

        // if ($inscription->getCourse()->getId() == $course->getId()) {
        //     return $inscription;
        // }

        $inscriptionsByCourse = $this->getInscriptions()->filter(function ($inscription) use ($course) {
            return $inscription->getCourse()->getId() == $course->getid();
        });

        $inscription = $inscriptionsByCourse->last();

        if ($inscription && $inscription->getCourse()->getId() == $course->getId()) {
            return $inscription;
        }

        // foreach ($this->getInscriptions() as $inscription) {
        //     $inscriptionCourseId = $inscription->getCourse()->getId();
        //     if ($inscriptionCourseId == $course->getId()) {
        //         return $inscription;
        //     }
        // }

        return null;
    }

    public function verifyInscriptionClass($courseClass)
    {

        $inscriptionsByCourse = $this->getInscriptionsClass()->filter(function ($inscriptionClass) use ($courseClass) {
            return $inscriptionClass->getCourseClass()->getId() == $courseClass->getid();
        });

        $inscription = $inscriptionsByCourse->last();

        if ($inscription && $inscription->getCourseClass()->getId() == $courseClass->getId()) {
            return $inscription;
        }

        return null;
    }

    /**
     * Set superior
     *
     * @param \UserBundle\Entity\User $superior
     *
     * @return User
     */
    public function setSuperior(\UserBundle\Entity\User $superior = null)
    {
        $this->superior = $superior;

        return $this;
    }

    /**
     * Get superior
     *
     * @return \UserBundle\Entity\User
     */
    public function getSuperior()
    {
        return $this->superior;
    }

    /**
     * Add permission
     *
     * @param \AppBundle\Entity\UserAccessControl $permission
     *
     * @return User
     */
    public function addPermission(\AppBundle\Entity\UserAccessControl $permission)
    {
        $permission->setUser($this);

        $this->permissions->add($permission);
        // $this->permissions[] = $permission;
    }

    /**
     * Remove permission
     *
     * @param \AppBundle\Entity\UserAccessControl $permission
     */
    public function removePermission(\AppBundle\Entity\UserAccessControl $permission)
    {
        $this->permissions->removeElement($permission);
    }

    /**
     * Set permissions
     *
     * @param string $permissions
     * @return User
     */
    public function setPermissions($permissions)
    {
        $this->permissions = $permissions;

        return $this;
    }

    /**
     * Get permissions
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getPermissions()
    {
        return $this->permissions;
    }

    public function imageBase64()
    {
        if ($this->photo) {
            if (strpos($this->photo, 'base64') !== false) {
                return $this->photo;
            } else {
                $path = 'uploads/user/photo/' . $this->photo;
                $type = pathinfo($path, PATHINFO_EXTENSION);
                $data = file_get_contents($path);
                $base64 = 'data:image/' . $type . ';base64,' . base64_encode($data);
                return $base64;
            }
        } else {
            return null;
        }
    }

    /**
     * Set usergroup
     *
     * @param integer $usergroup
     *
     * @return Client
     */
    public function setUsergroup($usergroup)
    {
        $this->usergroup = $usergroup;

        return $this;
    }

    /**
     * Get group
     *
     * @return integer
     */
    public function getUsergroup()
    {
        return $this->usergroup;
    }

    /**
     * Set accessGranted
     *
     * @param boolean $accessGranted
     * @return Answer
     */
    public function setAccessGranted($accessGranted)
    {
        $this->accessGranted = $accessGranted;

        return $this;
    }

    /**
     * Get accessGranted
     *
     * @return boolean
     */
    public function getAccessGranted()
    {
        return $this->accessGranted;
    }
}
