<?php

namespace IdeasBucket\Common\Utils;

class ArrayHelperTest extends \PHPUnit_Framework_TestCase
{
    public function testAccessible()
    {
        $this->assertTrue(ArrayHelper::accessible([]));
        $this->assertTrue(ArrayHelper::accessible([1, 2]));
        $this->assertTrue(ArrayHelper::accessible(['a' => 1, 'b' => 2]));
        $this->assertTrue(ArrayHelper::accessible(new Collection()));

        $this->assertFalse(ArrayHelper::accessible(null));
        $this->assertFalse(ArrayHelper::accessible('abc'));
        $this->assertFalse(ArrayHelper::accessible(new \stdClass()));
        $this->assertFalse(ArrayHelper::accessible((object) ['a' => 1, 'b' => 2]));
    }

    public function testAdd()
    {
        $array = ArrayHelper::add(['name' => 'Desk'], 'price', 100);
        $this->assertEquals(['name' => 'Desk', 'price' => 100], $array);
    }

    public function testCollapse()
    {
        $data = [['foo', 'bar'], ['baz']];
        $this->assertEquals(['foo', 'bar', 'baz'], ArrayHelper::collapse($data));
    }

    public function testDivide()
    {
        list($keys, $values) = ArrayHelper::divide(['name' => 'Desk']);
        $this->assertEquals(['name'], $keys);
        $this->assertEquals(['Desk'], $values);
    }

    public function testDot()
    {
        $array = ArrayHelper::dot(['foo' => ['bar' => 'baz']]);
        $this->assertEquals(['foo.bar' => 'baz'], $array);

        $array = ArrayHelper::dot([]);
        $this->assertEquals([], $array);

        $array = ArrayHelper::dot(['foo' => []]);
        $this->assertEquals(['foo' => []], $array);

        $array = ArrayHelper::dot(['foo' => ['bar' => []]]);
        $this->assertEquals(['foo.bar' => []], $array);
    }

    public function testExcept()
    {
        $array = ['name' => 'Desk', 'price' => 100];
        $array = ArrayHelper::except($array, ['price']);
        $this->assertEquals(['name' => 'Desk'], $array);
    }

    public function testExists()
    {
        $this->assertTrue(ArrayHelper::exists([1], 0));
        $this->assertTrue(ArrayHelper::exists([null], 0));
        $this->assertTrue(ArrayHelper::exists(['a' => 1], 'a'));
        $this->assertTrue(ArrayHelper::exists(['a' => null], 'a'));
        $this->assertTrue(ArrayHelper::exists(new Collection(['a' => null]), 'a'));

        $this->assertFalse(ArrayHelper::exists([1], 1));
        $this->assertFalse(ArrayHelper::exists([null], 1));
        $this->assertFalse(ArrayHelper::exists(['a' => 1], 0));
        $this->assertFalse(ArrayHelper::exists(new Collection(['a' => null]), 'b'));
    }

    public function testFirst()
    {
        $array = [100, 200, 300];

        $value = ArrayHelper::first($array, function ($value) {
            return $value >= 150;
        });

        $this->assertEquals(200, $value);
        $this->assertEquals(100, ArrayHelper::first($array));
    }

    public function testLast()
    {
        $array = [100, 200, 300];

        $last = ArrayHelper::last($array, function ($value) {
            return $value < 250;
        });
        $this->assertEquals(200, $last);

        $last = ArrayHelper::last($array, function ($value, $key) {
            return $key < 2;
        });
        $this->assertEquals(200, $last);

        $this->assertEquals(300, ArrayHelper::last($array));
    }

    public function testFlatten()
    {
        // Flat arrays are unaffected
        $array = ['#foo', '#bar', '#baz'];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten(['#foo', '#bar', '#baz']));

        // Nested arrays are flattened with existing flat items
        $array = [['#foo', '#bar'], '#baz'];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten($array));

        // Flattened array includes "null" items
        $array = [['#foo', null], '#baz', null];
        $this->assertEquals(['#foo', null, '#baz', null], ArrayHelper::flatten($array));

        // Sets of nested arrays are flattened
        $array = [['#foo', '#bar'], ['#baz']];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten($array));

        // Deeply nested arrays are flattened
        $array = [['#foo', ['#bar']], ['#baz']];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten($array));

        // Nested arrays are flattened alongside arrays
        $array = [new Collection(['#foo', '#bar']), ['#baz']];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten($array));

        // Nested arrays containing plain arrays are flattened
        $array = [new Collection(['#foo', ['#bar']]), ['#baz']];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten($array));

        // Nested arrays containing arrays are flattened
        $array = [['#foo', new Collection(['#bar'])], ['#baz']];
        $this->assertEquals(['#foo', '#bar', '#baz'], ArrayHelper::flatten($array));

        // Nested arrays containing arrays containing arrays are flattened
        $array = [['#foo', new Collection(['#bar', ['#zap']])], ['#baz']];
        $this->assertEquals(['#foo', '#bar', '#zap', '#baz'], ArrayHelper::flatten($array));
    }

    public function testFlattenWithDepth()
    {
        // No depth flattens recursively
        $array = [['#foo', ['#bar', ['#baz']]], '#zap'];
        $this->assertEquals(['#foo', '#bar', '#baz', '#zap'], ArrayHelper::flatten($array));

        // Specifying a depth only flattens to that depth
        $array = [['#foo', ['#bar', ['#baz']]], '#zap'];
        $this->assertEquals(['#foo', ['#bar', ['#baz']], '#zap'], ArrayHelper::flatten($array, 1));

        $array = [['#foo', ['#bar', ['#baz']]], '#zap'];
        $this->assertEquals(['#foo', '#bar', ['#baz'], '#zap'], ArrayHelper::flatten($array, 2));
    }

    public function testGet()
    {
        $array = ['products.desk' => ['price' => 100]];
        $this->assertEquals(['price' => 100], ArrayHelper::get($array, 'products.desk'));

        $array = ['products' => ['desk' => ['price' => 100]]];
        $value = ArrayHelper::get($array, 'products.desk');
        $this->assertEquals(['price' => 100], $value);

        // Test null array values
        $array = ['foo' => null, 'bar' => ['baz' => null]];
        $this->assertNull(ArrayHelper::get($array, 'foo', 'default'));
        $this->assertNull(ArrayHelper::get($array, 'bar.baz', 'default'));

        // Test direct ArrayAccess object
        $array = ['products' => ['desk' => ['price' => 100]]];
        $arrayAccessObject = new \ArrayObject($array);
        $value = ArrayHelper::get($arrayAccessObject, 'products.desk');
        $this->assertEquals(['price' => 100], $value);

        // Test array containing ArrayAccess object
        $arrayAccessChild = new \ArrayObject(['products' => ['desk' => ['price' => 100]]]);
        $array = ['child' => $arrayAccessChild];
        $value = ArrayHelper::get($array, 'child.products.desk');
        $this->assertEquals(['price' => 100], $value);

        // Test array containing multiple nested ArrayAccess objects
        $arrayAccessChild = new \ArrayObject(['products' => ['desk' => ['price' => 100]]]);
        $arrayAccessParent = new \ArrayObject(['child' => $arrayAccessChild]);
        $array = ['parent' => $arrayAccessParent];
        $value = ArrayHelper::get($array, 'parent.child.products.desk');
        $this->assertEquals(['price' => 100], $value);

        // Test missing ArrayAccess object field
        $arrayAccessChild = new \ArrayObject(['products' => ['desk' => ['price' => 100]]]);
        $arrayAccessParent = new \ArrayObject(['child' => $arrayAccessChild]);
        $array = ['parent' => $arrayAccessParent];
        $value = ArrayHelper::get($array, 'parent.child.desk');
        $this->assertNull($value);

        // Test missing ArrayAccess object field
        $arrayAccessObject = new \ArrayObject(['products' => ['desk' => null]]);
        $array = ['parent' => $arrayAccessObject];
        $value = ArrayHelper::get($array, 'parent.products.desk.price');
        $this->assertNull($value);

        // Test null ArrayAccess object fields
        $array = new \ArrayObject(['foo' => null, 'bar' => new \ArrayObject(['baz' => null])]);
        $this->assertNull(ArrayHelper::get($array, 'foo', 'default'));
        $this->assertNull(ArrayHelper::get($array, 'bar.baz', 'default'));

        // Test null key returns the whole array
        $array = ['foo', 'bar'];
        $this->assertEquals($array, ArrayHelper::get($array, null));

        // Test $array not an array
        $this->assertSame('default', ArrayHelper::get(null, 'foo', 'default'));
        $this->assertSame('default', ArrayHelper::get(false, 'foo', 'default'));

        // Test $array not an array and key is null
        $this->assertSame('default', ArrayHelper::get(null, null, 'default'));

        // Test $array is empty and key is null
        $this->assertSame([], ArrayHelper::get([], null));
        $this->assertSame([], ArrayHelper::get([], null, 'default'));
    }

    public function testHas()
    {
        $array = ['products.desk' => ['price' => 100]];
        $this->assertTrue(ArrayHelper::has($array, 'products.desk'));

        $array = ['products' => ['desk' => ['price' => 100]]];
        $this->assertTrue(ArrayHelper::has($array, 'products.desk'));
        $this->assertTrue(ArrayHelper::has($array, 'products.desk.price'));
        $this->assertFalse(ArrayHelper::has($array, 'products.foo'));
        $this->assertFalse(ArrayHelper::has($array, 'products.desk.foo'));

        $array = ['foo' => null, 'bar' => ['baz' => null]];
        $this->assertTrue(ArrayHelper::has($array, 'foo'));
        $this->assertTrue(ArrayHelper::has($array, 'bar.baz'));

        $array = new \ArrayObject(['foo' => 10, 'bar' => new \ArrayObject(['baz' => 10])]);
        $this->assertTrue(ArrayHelper::has($array, 'foo'));
        $this->assertTrue(ArrayHelper::has($array, 'bar'));
        $this->assertTrue(ArrayHelper::has($array, 'bar.baz'));
        $this->assertFalse(ArrayHelper::has($array, 'xxx'));
        $this->assertFalse(ArrayHelper::has($array, 'xxx.yyy'));
        $this->assertFalse(ArrayHelper::has($array, 'foo.xxx'));
        $this->assertFalse(ArrayHelper::has($array, 'bar.xxx'));

        $array = new \ArrayObject(['foo' => null, 'bar' => new \ArrayObject(['baz' => null])]);
        $this->assertTrue(ArrayHelper::has($array, 'foo'));
        $this->assertTrue(ArrayHelper::has($array, 'bar.baz'));

        $array = ['foo', 'bar'];
        $this->assertFalse(ArrayHelper::has($array, null));

        $this->assertFalse(ArrayHelper::has(null, 'foo'));
        $this->assertFalse(ArrayHelper::has(false, 'foo'));

        $this->assertFalse(ArrayHelper::has(null, null));
        $this->assertFalse(ArrayHelper::has([], null));

        $array = ['products' => ['desk' => ['price' => 100]]];
        $this->assertTrue(ArrayHelper::has($array, ['products.desk']));
        $this->assertTrue(ArrayHelper::has($array, ['products.desk', 'products.desk.price']));
        $this->assertTrue(ArrayHelper::has($array, ['products', 'products']));
        $this->assertFalse(ArrayHelper::has($array, ['foo']));
        $this->assertFalse(ArrayHelper::has($array, []));
        $this->assertFalse(ArrayHelper::has($array, ['products.desk', 'products.price']));

        $this->assertFalse(ArrayHelper::has([], [null]));
        $this->assertFalse(ArrayHelper::has(null, [null]));
    }

    public function testIsAssoc()
    {
        $this->assertTrue(ArrayHelper::isAssoc(['a' => 'a', 0 => 'b']));
        $this->assertTrue(ArrayHelper::isAssoc([1 => 'a', 0 => 'b']));
        $this->assertTrue(ArrayHelper::isAssoc([1 => 'a', 2 => 'b']));
        $this->assertFalse(ArrayHelper::isAssoc([0 => 'a', 1 => 'b']));
        $this->assertFalse(ArrayHelper::isAssoc(['a', 'b']));
    }

    public function testOnly()
    {
        $array = ['name' => 'Desk', 'price' => 100, 'orders' => 10];
        $array = ArrayHelper::only($array, ['name', 'price']);
        $this->assertEquals(['name' => 'Desk', 'price' => 100], $array);
    }

    public function testPluck()
    {
        $array = [
            ['developer' => ['name' => 'Taylor']],
            ['developer' => ['name' => 'Abigail']],
        ];

        $array = ArrayHelper::pluck($array, 'developer.name');

        $this->assertEquals(['Taylor', 'Abigail'], $array);
    }

    public function testPluckWithKeys()
    {
        $array = [
            ['name' => 'Taylor', 'role' => 'developer'],
            ['name' => 'Abigail', 'role' => 'developer'],
        ];

        $test1 = ArrayHelper::pluck($array, 'role', 'name');
        $test2 = ArrayHelper::pluck($array, null, 'name');

        $this->assertEquals([
            'Taylor'  => 'developer',
            'Abigail' => 'developer',
        ], $test1);

        $this->assertEquals([
            'Taylor'  => ['name' => 'Taylor', 'role' => 'developer'],
            'Abigail' => ['name' => 'Abigail', 'role' => 'developer'],
        ], $test2);
    }

    public function testPrepend()
    {
        $array = ArrayHelper::prepend(['one', 'two', 'three', 'four'], 'zero');
        $this->assertEquals(['zero', 'one', 'two', 'three', 'four'], $array);

        $array = ArrayHelper::prepend(['one' => 1, 'two' => 2], 0, 'zero');
        $this->assertEquals(['zero' => 0, 'one' => 1, 'two' => 2], $array);
    }

    public function testPull()
    {
        $array = ['name' => 'Desk', 'price' => 100];
        $name = ArrayHelper::pull($array, 'name');
        $this->assertEquals('Desk', $name);
        $this->assertEquals(['price' => 100], $array);

        // Only works on first level keys
        $array = ['joe@example.com' => 'Joe', 'jane@localhost' => 'Jane'];
        $name = ArrayHelper::pull($array, 'joe@example.com');
        $this->assertEquals('Joe', $name);
        $this->assertEquals(['jane@localhost' => 'Jane'], $array);

        // Does not work for nested keys
        $array = ['emails' => ['joe@example.com' => 'Joe', 'jane@localhost' => 'Jane']];
        $name = ArrayHelper::pull($array, 'emails.joe@example.com');
        $this->assertEquals(null, $name);
        $this->assertEquals(['emails' => ['joe@example.com' => 'Joe', 'jane@localhost' => 'Jane']], $array);
    }

    public function testSet()
    {
        $array = ['products' => ['desk' => ['price' => 100]]];
        ArrayHelper::set($array, 'products.desk.price', 200);
        $this->assertEquals(['products' => ['desk' => ['price' => 200]]], $array);
    }

    public function testSort()
    {
        $unsorted = [
            ['name' => 'Desk'],
            ['name' => 'Chair'],
        ];

        $expected = [
            ['name' => 'Chair'],
            ['name' => 'Desk'],
        ];

        // sort with closure
        $sortedWithClosure = array_values(ArrayHelper::sort($unsorted, function ($value) {
            return $value['name'];
        }));
        $this->assertEquals($expected, $sortedWithClosure);

        // sort with dot notation
        $sortedWithDotNotation = array_values(ArrayHelper::sort($unsorted, 'name'));
        $this->assertEquals($expected, $sortedWithDotNotation);
    }

    public function testSortRecursive()
    {
        $array = [
            'users' => [
                [
                    // should sort associative arrays by keys
                    'name' => 'joe',
                    'mail' => 'joe@example.com',
                    // should sort deeply nested arrays
                    'numbers' => [2, 1, 0],
                ],
                [
                    'name' => 'jane',
                    'age'  => 25,
                ],
            ],
            'repositories' => [
                // should use weird `sort()` behavior on arrays of arrays
                ['id' => 1],
                ['id' => 0],
            ],
            // should sort non-associative arrays by value
            20 => [2, 1, 0],
            30 => [
                // should sort non-incrementing numerical keys by keys
                2 => 'a',
                1 => 'b',
                0 => 'c',
            ],
        ];

        $expect = [
            20 => [0, 1, 2],
            30 => [
                0 => 'c',
                1 => 'b',
                2 => 'a',
            ],
            'repositories' => [
                ['id' => 0],
                ['id' => 1],
            ],
            'users' => [
                [
                    'age'  => 25,
                    'name' => 'jane',
                ],
                [
                    'mail'    => 'joe@example.com',
                    'name'    => 'joe',
                    'numbers' => [0, 1, 2],
                ],
            ],
        ];

        $this->assertEquals($expect, ArrayHelper::sortRecursive($array));
    }

    public function testWhere()
    {
        $array = [100, '200', 300, '400', 500];

        $array = ArrayHelper::where($array, function ($value, $key) {
            return is_string($value);
        });

        $this->assertEquals([1 => 200, 3 => 400], $array);
    }

    public function testWhereKey()
    {
        $array = ['10' => 1, 'foo' => 3, 20 => 2];

        $array = ArrayHelper::where($array, function ($value, $key) {
            return is_numeric($key);
        });

        $this->assertEquals(['10' => 1, 20 => 2], $array);
    }

    public function testForget()
    {
        $array = ['products' => ['desk' => ['price' => 100]]];
        ArrayHelper::forget($array, null);
        $this->assertEquals(['products' => ['desk' => ['price' => 100]]], $array);

        $array = ['products' => ['desk' => ['price' => 100]]];
        ArrayHelper::forget($array, []);
        $this->assertEquals(['products' => ['desk' => ['price' => 100]]], $array);

        $array = ['products' => ['desk' => ['price' => 100]]];
        ArrayHelper::forget($array, 'products.desk');
        $this->assertEquals(['products' => []], $array);

        $array = ['products' => ['desk' => ['price' => 100]]];
        ArrayHelper::forget($array, 'products.desk.price');
        $this->assertEquals(['products' => ['desk' => []]], $array);

        $array = ['products' => ['desk' => ['price' => 100]]];
        ArrayHelper::forget($array, 'products.final.price');
        $this->assertEquals(['products' => ['desk' => ['price' => 100]]], $array);

        $array = ['shop' => ['cart' => [150 => 0]]];
        ArrayHelper::forget($array, 'shop.final.cart');
        $this->assertEquals(['shop' => ['cart' => [150 => 0]]], $array);

        $array = ['products' => ['desk' => ['price' => ['original' => 50, 'taxes' => 60]]]];
        ArrayHelper::forget($array, 'products.desk.price.taxes');
        $this->assertEquals(['products' => ['desk' => ['price' => ['original' => 50]]]], $array);

        $array = ['products' => ['desk' => ['price' => ['original' => 50, 'taxes' => 60]]]];
        ArrayHelper::forget($array, 'products.desk.final.taxes');
        $this->assertEquals(['products' => ['desk' => ['price' => ['original' => 50, 'taxes' => 60]]]], $array);

        $array = ['products' => ['desk' => ['price' => 50], null => 'something']];
        ArrayHelper::forget($array, ['products.amount.all', 'products.desk.price']);
        $this->assertEquals(['products' => ['desk' => [], null => 'something']], $array);

        // Only works on first level keys
        $array = ['joe@example.com' => 'Joe', 'jane@example.com' => 'Jane'];
        ArrayHelper::forget($array, 'joe@example.com');
        $this->assertEquals(['jane@example.com' => 'Jane'], $array);

        // Does not work for nested keys
        $array = ['emails' => ['joe@example.com' => ['name' => 'Joe'], 'jane@localhost' => ['name' => 'Jane']]];
        ArrayHelper::forget($array, ['emails.joe@example.com', 'emails.jane@localhost']);
        $this->assertEquals(['emails' => ['joe@example.com' => ['name' => 'Joe']]], $array);
    }
}
